import { makeObservable, action } from 'mobx';
import cloneDeep from 'lodash/cloneDeep';
import { AbstractStrategy } from '../AbstractStrategy';
import { ChangeStrategy } from '../types';

export class ChangeOffStrategy extends AbstractStrategy implements ChangeStrategy {
  public constructor() {
    super();
    makeObservable(this, {
      makePathInValueAsTree: action.bound,
    });
  }

  public applyForLeaf(id: number, path: number[]) {
    this.makePathInValueAsTree(id, path);
    this.recursiveLeafApplying(id, path);
  }

  public recursiveLeafApplying(id: number, path: number[]) {
    const { getById } = this.tree;
    const category = getById(id);

    if (!category) {
      return;
    }

    category.check.off();
    if (category.items) {
      category.items.forEach((categoryId) => {
        this.recursiveLeafApplying(categoryId, [...path, categoryId]);
      });
    }
  }

  public makePathInValueAsTree(id: number, path: number[]) {
    const newValueAsTree = cloneDeep(this.tree.valueAsTree.current);
    const pathForRemove = path.slice(0, -1);
    let node = newValueAsTree;
    pathForRemove.forEach((id) => {
      node = node[id];
    });
    delete node[id];
    this.tree.valueAsTree.current = newValueAsTree;
  }

  public canContinueBranchChange(id: number, path: number[]) {
    const lastPathId = path[path.length - 1];
    const category = this.tree.getById(lastPathId);
    if (!category || !category.parentId) {
      return false;
    }

    const siblingCategoryIds = ((this.tree.getById(category.parentId) || {}).items || []).filter(
      (categoryId) => categoryId !== id,
    );

    const isSomeSiblingChecked = siblingCategoryIds
      .filter((categoryId) => this.tree.getById(categoryId))
      .some((categoryId) => this.tree.getById(categoryId)!.check.state);

    if (isSomeSiblingChecked) {
      return false;
    }

    return true;
  }

  public applyForBranch(id: number, path: number[]) {
    if (!this.canInteractWith(path)) {
      return;
    }

    this.applyForLeaf(id, path);

    if (!this.canContinueBranchChange(id, path)) {
      return;
    }

    const category = this.tree.getById(id);
    if (!category) {
      return;
    }
    this.applyForBranch(category.parentId as number, path.slice(0, -1));
  }
}
