import { makeObservable, observable, computed, action } from 'mobx';
import isEqual from 'lodash/isEqual';
import last from 'lodash/last';
import {
  ValueAsTree,
  Tree as ITree,
  CategoriesObject,
  ById,
  CategoryDTO,
  Category as ICategory,
} from '../../../types';
import { getPathFromHighestParent, filterRootIds } from '../../../utils';
import { ChangeStrategy, MultiBranchStrategy } from './ChangeStrategies';
import { getFiniteChildren } from '../../Provider/Provider.utils';
import { AbstractConfigurator } from '../AbstractConfigurator';
import { Configurator } from './Configurator';
import { BoolState } from '../BoolState';
import { Category } from '../Category';

export class Tree implements ITree {
  public constructor(
    changeStrategy: ChangeStrategy = new MultiBranchStrategy(),
    configurator: AbstractConfigurator<[CategoriesObject]> = new Configurator(),
  ) {
    makeObservable(this, {
      byId: observable,
      root: observable,
      highlightPath: observable,
      valueAsTree: observable,
      lastHighlighted: computed,
      expanded: computed,
      finiteSelected: computed,
      rehighlightPath: action.bound,
      selectCategory: action.bound,
      setup: action.bound,
      changeLeafValue: action.bound,
      changeBranchValue: action.bound,
      reset: action.bound,
    });

    this.setChangeStrategy(changeStrategy);
    this.setConfigurator(configurator);
    this.initCategoryDTOById = this.initCategoryDTOById.bind(this);
    this.getById = this.getById.bind(this);
  }

  public loading = new BoolState(false);

  private changeStrategy: ChangeStrategy;

  private configurator: AbstractConfigurator<[CategoriesObject]>;

  public byId: ById = {};

  public jsonById: Record<number, CategoryDTO | undefined> = {};

  public root: number[] = [];

  public highlightPath: number[] = [];

  public valueAsTree: ValueAsTree = {
    current: {},
    initial: {},
  };

  public getById(id: number): ICategory | undefined {
    if (this.byId[id]) {
      return this.byId[id];
    }

    return this.initCategoryDTOById(id);
  }

  public initCategoryDTOById(id: number): ICategory | undefined {
    const categoryDTO = this.jsonById[id];
    if (!categoryDTO) {
      return;
    }

    const category = (this.byId[id] = new Category(categoryDTO));
    delete this.jsonById[id];
    return category;
  }

  public get finiteSelected() {
    return filterRootIds(this.getById, getFiniteChildren(this.valueAsTree.current));
  }

  public get lastHighlighted() {
    return this.highlightPath[this.highlightPath.length - 1];
  }

  public get expanded() {
    return this.highlightPath.filter((id) => {
      const category = this.getById(id);
      return category && category.canExpand;
    });
  }

  public highlightId(id: number | undefined, isHighlighted: boolean) {
    if (id == null) {
      return;
    }

    const category = this.getById(id);
    if (category) {
      category.highlighting.set(isHighlighted);
    }
  }

  public wasIdAddedToPath(path: number[]) {
    return (
      path.length === this.highlightPath.length + 1 &&
      isEqual(path.slice(0, -1), this.highlightPath)
    );
  }

  public highlightAddedId(path: number[]) {
    this.highlightId(last(path), true);
  }

  public wasIdRemovedFromPath(path: number[]) {
    return (
      path.length === this.highlightPath.length - 1 &&
      isEqual(path, this.highlightPath.slice(0, -1))
    );
  }

  public highlightRemovedId() {
    this.highlightId(last(this.highlightPath), false);
  }

  public wasIdReplacedInPath(path: number[]) {
    return (
      path.length === this.highlightPath.length &&
      isEqual(path.slice(0, -1), this.highlightPath.slice(0, -1))
    );
  }

  public highlightReplacedId(path: number[]) {
    this.highlightId(last(path), true);
    this.highlightId(last(this.highlightPath), false);
  }

  public tryToOptimizeRehighlight(path: number[]): boolean {
    if (isEqual(path, this.highlightPath)) {
      return true;
    }

    if (this.wasIdReplacedInPath(path)) {
      this.highlightReplacedId(path);
      return true;
    }

    if (this.wasIdAddedToPath(path)) {
      this.highlightAddedId(path);
      return true;
    }

    if (this.wasIdRemovedFromPath(path)) {
      this.highlightRemovedId();
      return true;
    }

    return false;
  }

  public rehighlightPath(path: number[]) {
    this.highlightPath.forEach((id) => {
      const category = this.getById(id);
      if (category) {
        category.highlighting.off();
      }
    });
    path.forEach((id) => {
      const category = this.getById(id);
      if (category) {
        category.highlighting.on();
      }
    });
  }

  public selectCategory(id: number) {
    const highlightPath = getPathFromHighestParent(this.getById, id);
    if (!this.tryToOptimizeRehighlight(highlightPath)) {
      this.rehighlightPath(highlightPath);
    }
    this.highlightPath = highlightPath;
  }

  public changeLeafValue(
    id: number,
    value: boolean,
    path = getPathFromHighestParent(this.getById, id),
  ) {
    this.changeStrategy.applyForLeaf(id, path, value);
  }

  public changeBranchValue(
    id: number,
    value: boolean,
    path = getPathFromHighestParent(this.getById, id),
  ) {
    this.changeStrategy.applyForBranch(id, path, value);
  }

  public setChangeStrategy(strategy: ChangeStrategy) {
    this.changeStrategy = strategy;
    this.changeStrategy.setTree(this);
  }

  public setConfigurator(configurator: AbstractConfigurator<[CategoriesObject]>) {
    this.configurator = configurator;
    this.configurator.setConfigurable(this);
  }

  public setup(categoriesObject: CategoriesObject) {
    this.configurator.setup(categoriesObject);
  }

  public reset() {
    this.byId = {};
    this.jsonById = {};
    this.root = [];
    this.highlightPath = [];
    this.loading.off();
    this.valueAsTree = {
      current: {},
      initial: {},
    };
  }
}
