import React, { FC, useRef, useEffect } from 'react';
import cx from 'classnames';
import { Overlay } from 'components/Overlay';
import { IFrameProps } from './IFrame.types';
import css from './IFrame.module.css';
import { IFRAME_INNER_MARGIN, PADDING } from './IFrame.constants';

export const IFrame: FC<IFrameProps> = (props) => {
  const { srcDoc, isLoading } = props;
  const prevIframeHeight = useRef(0);
  const nodeRef = useRef<HTMLDivElement>(null);
  const iframeRef = useRef<HTMLIFrameElement>(null);

  const className = cx(css.IFrame, props.className, {
    [css.IFrame_hidden]: isLoading,
  });

  useEffect(() => {
    const intervalId = setInterval(() => {
      const iframeNode = iframeRef.current;
      const node = nodeRef.current;
      if (!iframeNode || !iframeNode.contentDocument || !iframeNode.contentDocument.body || !node) {
        return;
      }

      const currentHeight =
        iframeNode.contentDocument.body.clientHeight + IFRAME_INNER_MARGIN + PADDING;
      if (currentHeight !== prevIframeHeight.current) {
        node.style.height = currentHeight + 'px';
        iframeNode.style.height = currentHeight + 'px';
      }
      prevIframeHeight.current = currentHeight;
    }, 300);

    return () => {
      clearInterval(intervalId);
    };
  }, [srcDoc]);

  return (
    <div className={className} ref={nodeRef}>
      <Overlay color="#fff" hasSpinner={isLoading} display={isLoading} />
      <iframe className={css.IFrame__control} scrolling="no" srcDoc={srcDoc} ref={iframeRef} />
    </div>
  );
};
