import React, { FC, useState, useRef, useEffect } from 'react';
import cx from 'classnames';
import Icon from '@crm/components/dist/lego2/Icon';
import Tooltip from '@crm/components/dist/lego2/Tooltip';
import { Direction } from '@crm/components/dist/lego2/withOldDirections';
import { RateButtonProps } from './RateButton.types';
import css from './RateButton.module.css';
import { TOOLTIP_DELAY } from './RateButton.constants';

const direction: Direction[] = ['bottom-right', 'bottom-center'];

export const RateButton: FC<RateButtonProps> = (props) => {
  const { type = 'like', onClick, clicked, disabled, onWantToComment, ...restProps } = props;
  const rootRef = useRef<HTMLDivElement>(null);
  const buttonRef = useRef<HTMLButtonElement>(null);
  const timerId = useRef<number>();
  const [isTooltipVisible, setTooltipVisible] = useState(false);

  const handleClose = () => {
    setTooltipVisible(false);
  };

  useEffect(
    () => () => {
      if (timerId.current) {
        window.clearTimeout(timerId.current);
      }
    },
    [],
  );

  const handleClick = () => {
    if (disabled) {
      return;
    }

    if (clicked) {
      setTooltipVisible(false);
    } else {
      setTooltipVisible(true);
    }

    if (props.onClick) {
      props.onClick();
    }
  };

  const handleWantToCommentClick = () => {
    if (onWantToComment) {
      onWantToComment(type);
    }
  };

  const handleMouseLeave = () => {
    timerId.current = window.setTimeout(() => {
      setTooltipVisible(false);
    }, TOOLTIP_DELAY);
  };

  const handleMouseEnter = () => {
    if (timerId.current != null) {
      window.clearTimeout(timerId.current);
    }

    if (clicked && !disabled) {
      setTooltipVisible(true);
    }
  };

  const className = cx(css.RateButton, {
    [css.RateButton_clicked]: clicked,
    [css.RateButton_type_dislike]: type === 'dislike',
  });

  return (
    <div
      className={props.className}
      ref={rootRef}
      onMouseLeave={handleMouseLeave}
      onMouseEnter={handleMouseEnter}
    >
      <button
        onClick={handleClick}
        {...restProps}
        ref={buttonRef}
        className={className}
        disabled={disabled}
      >
        <Icon className={css.RateButton__icon} svg="like" svgSize="l" />
      </button>
      <Tooltip
        className={css.RateButton__tooltip}
        view="action"
        anchor={buttonRef}
        visible={isTooltipVisible}
        scope={rootRef}
        hasTail
        direction={direction}
        onClose={handleClose}
      >
        <div onClick={handleWantToCommentClick}>Оставить комментарий</div>
      </Tooltip>
    </div>
  );
};
