import React, { FC, useState, useEffect } from 'react';
import cx from 'classnames';
import { observer } from 'mobx-react-lite';
import { config } from 'services/Config';
import { Overlay } from 'components/Overlay';
import Button from '@crm/components/dist/lego2/Button';
import { FullScreen } from 'components/FullScreen';
import { IFrame } from './IFrame';
import { ButtonPanel } from './ButtonPanel';
import { LeaveComment } from './LeaveComment';
import { TipProps } from './Tip.types';
import css from './Tip.module.css';
import { useContextState } from '../../State';

export const Tip: FC<TipProps> = observer((props) => {
  const className = cx(css.Tip, props.className);
  const { onToggle, onLikeClick, onDislikeClick, onTipOpen } = props;
  const [isFullScreen, setFullScreen] = useState(false);
  const { categoryTipsButtonExperiment } = config.value.features;
  const [isTipOpen, setIsTipOpen] = useState(false);
  const hasOpenTipButton = categoryTipsButtonExperiment && !isTipOpen;

  const state = useContextState();
  const { tip, tree } = state;
  const { html, loading } = tip;
  const { lastHighlighted } = tree as { lastHighlighted: number };
  const { rating } = tip;
  const { commentForm } = rating;

  useEffect(() => {
    setIsTipOpen(false);
  }, [lastHighlighted]);

  const handleToggle = (expand: boolean) => {
    setFullScreen(expand);
    if (onToggle) {
      onToggle(expand);
    }
  };

  const handleLikeClick = () => {
    rating.like(lastHighlighted);
    if (onLikeClick) {
      onLikeClick();
    }
  };

  const handleDislikeClick = () => {
    rating.dislike(lastHighlighted);
    if (onDislikeClick) {
      onDislikeClick();
    }
  };

  const handleOpenTipClick = () => {
    setIsTipOpen(true);
    if (onTipOpen) {
      onTipOpen();
    }
  };

  const renderedButtonPanel = (
    <ButtonPanel
      isExpanded={isFullScreen}
      onExpandToggle={handleToggle}
      onLikeClick={handleLikeClick}
      onDislikeClick={handleDislikeClick}
      onWantToComment={commentForm.openess.on}
      rating={rating.byId[lastHighlighted]}
      disabled={!html || hasOpenTipButton}
    />
  );

  const renderedTopLine = (
    <div className={css.Tip__topLine}>
      <span className={css.Tip__topLineTitle}>Подсказка</span>
      {renderedButtonPanel}
    </div>
  );

  const emptyTip = !html ? (
    <div className={css.Tip__emptyTip}>
      <div className={css.Tip__text}>
        <p className={css.Tip__textRow}>Нет подсказок для данной категории</p>
      </div>
    </div>
  ) : null;

  const renderedTip = html ? (
    <IFrame srcDoc={html} isLoading={loading.state} className={css.Tip__iframe} />
  ) : null;

  const openTipButton =
    html && !loading.state ? (
      <Button view="action" onClick={handleOpenTipClick}>
        Показать подсказку
      </Button>
    ) : null;

  const handleTipOverlayClick = () => {
    setFullScreen(false);
  };

  return (
    <div className={className} role="note">
      {renderedTopLine}
      {emptyTip}
      {hasOpenTipButton ? openTipButton : renderedTip}

      {html && isFullScreen && (
        <FullScreen>
          <Overlay
            display
            hasSpinner={false}
            color="rgba(0, 0, 0, .4)"
            onClick={handleTipOverlayClick}
          />
          <div className={css.Tip__fullScreenWrap}>
            {renderedTopLine}
            <div className={css.Tip__fullScreenIframeWrap}>
              <IFrame srcDoc={html} isLoading={loading.state} />
            </div>
          </div>
        </FullScreen>
      )}
      <LeaveComment categoryId={lastHighlighted} commentForm={commentForm} />
    </div>
  );
});
