import { ComponentType } from 'react';
import { Subject } from 'rxjs';
import { AlertDTO } from 'types/dto/AlertDTO';
import { ChangeStrategy } from './State/defaultStores/Tree/ChangeStrategies';
import { BoolState } from './State/defaultStores/BoolState';

export type ChangeStrategyType = 'one-branch' | 'multi-branch';

export interface EmittingPayload extends Partial<TargetMeta> {
  categoryId?: number;
  categoryIdHasAlert?: number;
  hasAlert?: boolean;
  block?: TabType | 'categorization';
  order?: number;
  resultsCount?: number;
  text?: string;
  hasTip?: boolean;
  commentId?: number;
}

export interface ExternalData {
  targetMeta: TargetMeta;
  previewComponent?: ComponentType;
  loadAlerts?: AlertLoadHandler;
  save?(value: { id: number }[]): void;
  search?(
    text: string,
  ): Promise<{
    resultIds: number[];
    highlightRangesById: TextHighlightingById;
  }>;
  loadTip?(id: number): Promise<string>;
  load?(): Promise<CategoriesObject>;
  commentSubmit?: CommentFormExternalHandler;
  changeStrategy?: ChangeStrategyType;
  emitter?: StoreEmitter;
}

export type TabType = 'tree' | 'search' | 'selected';

export interface EmittingPayload {
  categoryId?: number;
}

export interface TargetMeta {
  type: string;
  id: number;
  description?: string;
}

export interface Resetable {
  reset(): void;
}

export interface RunsReactions {
  runReactions(): (() => void)[];
}

export interface HasLoadingState {
  loading: BoolState;
}

export interface HasErrorState {
  error: BoolState;
}

export interface HasOpenessState {
  openess: BoolState;
}

export interface HasStores<TRecord extends {}> {
  setStores(stores: TRecord): void;
}

export interface SetsExternalHandler<HandlerType extends Function> {
  setExternalHandler(externalHandler: HandlerType): void;
}

export interface Configurable<TParams extends unknown[]> {
  setup(...parameters: TParams): void;
}

export interface SearchExternalHandler {
  (text: string): Promise<{
    resultIds: number[];
    highlightRangesById: TextHighlightingById;
  }>;
}

export interface Search
  extends Resetable,
    HasStores<{
      main: Store;
    }>,
    SetsExternalHandler<SearchExternalHandler>,
    RunsReactions,
    HasLoadingState,
    HasErrorState {
  handler(text: string): void;
  text: string;
  setText(text: string): void;
  resultIds: number[];
}

export type TextHighlightingById = Record<number, Range[] | undefined>;

export interface TextHighlighting extends Resetable {
  byId: TextHighlightingById;
  setById(byId: TextHighlightingById): void;
}

export type ById = Record<number, Category>;
export type GetById<T = Category> = (id: number) => T | undefined;

export interface Category extends CategoryDTO {
  highlighting: BoolState;
  check: BoolState;
  expanding: BoolState;
  canExpand: boolean;
}

export interface Tree extends Resetable, HasLoadingState, Configurable<[CategoriesObject]> {
  byId: ById;
  getById: (id: number) => Category | undefined;
  root: number[];
  highlightPath: number[];
  lastHighlighted?: number;
  valueAsTree: ValueAsTree;
  finiteSelected: number[];
  expanded: number[];
  selectCategory(id: number): void;
  changeLeafValue(id: number, value: boolean, path?: number[]): void;
  changeBranchValue(id: number, value: boolean, path?: number[]): void;
  setChangeStrategy(changeStrategy: ChangeStrategy): void;
}

export interface Tabs extends Resetable {
  current: TabType;
  previous?: TabType;
  go(tab: TabType): void;
  back(): void;
}

export interface TipExternalHandler {
  (id: number): Promise<string>;
}

export interface Tip
  extends Resetable,
    HasStores<{
      main: Store;
    }>,
    HasLoadingState,
    SetsExternalHandler<TipExternalHandler>,
    RunsReactions {
  html?: string;
  load(id: number): void;
  rating: Rating;
}

export interface CommentFormExternalHandler {
  (payload: CommentFormExternalHandlerPayload): Promise<{ id: number }> | void;
}

export interface CommentForm
  extends SetsExternalHandler<CommentFormExternalHandler>,
    HasStores<{
      main: Store;
      rating: Rating;
    }>,
    HasLoadingState,
    HasOpenessState,
    Resetable {
  comment(categoryId: number, text: string): Promise<{ id: number }> | void;
}

export interface CommentFormExternalHandlerPayload {
  id: number;
  text: string;
  rate: boolean | undefined;
}

export interface Rating
  extends Resetable,
    HasStores<{
      main: Store;
    }>,
    SetsExternalHandler<CommentFormExternalHandler>,
    RunsReactions {
  byId: Record<number, boolean | undefined>;
  like(id: number): void;
  dislike(id: number): void;
  commentForm: CommentForm;
}

export interface AlertStore
  extends SetsExternalHandler<AlertLoadHandler>,
    Resetable,
    RunsReactions,
    HasStores<{
      tree: Tree;
      main: Store;
    }> {
  load(categoryIds: number[]): void;
  expanding: BoolState;
  byCategoryId: Map<number, AlertDTO | undefined>;
  visibleAlert: AlertDTO | undefined;
  hasRelatedAlerts: boolean;
  relatedAlerts: AlertDTO[];
}

export type SaveHandler = (values: { id: number }[]) => void;
export type LoadHandler = () => Promise<CategoriesObject>;
export type StoreEmitter = Subject<{
  event: string;
  payload?: TargetMeta & EmittingPayload;
}>;
export type AlertLoadHandler = (categoryIds: number[]) => Promise<AlertDTO[]>;

export interface Store
  extends RunsReactions,
    Resetable,
    HasOpenessState,
    Configurable<[ExternalData?]> {
  targetMeta?: TargetMeta;
  setEmitter(emitter: StoreEmitter): void;
  emit<ExtraData extends EmittingPayload>(event: string, extraData?: ExtraData): void;

  tabs: Tabs;
  search: Search;
  tree: Tree;
  tip: Tip;
  textHighlighting: TextHighlighting;
  alertStore: AlertStore;
  setLoadHandler(loadHandler: LoadHandler): void;
  hasChanges: boolean;
  save(): void;
  setSaveHandler(saveHandler: SaveHandler): void;
  open(externalData?: ExternalData): void;
  close(): void;

  fullness: BoolState;
  previewComponent: ComponentType;
}

export interface CategoryDTO {
  id: number;
  parentId?: number;
  name: string;
  isLeaf: boolean;
  items?: number[];
}

export type DenormalizedCategory = Omit<CategoryDTO, 'items'> & { items: DenormalizedCategory[] };

export interface ValueAsTreeNode {
  [id: string]: ValueAsTreeNode;
}

export interface ValueAsTree {
  current: ValueAsTreeNode;
  initial: ValueAsTreeNode;
}

export interface CategoriesObject {
  byId: Record<number, CategoryDTO>;
  root: number[];
  highlightPath?: number[];
  valueAsTree: ValueAsTreeNode;
}

export type Range = [number, number];
