import * as React from 'react';
import cx from 'classnames';
import { FloydChatData } from 'modules/chat/types';
import { connect } from 'react-redux';
import css from './Chat.scss';
import FakeChat from './FakeChat';

interface ChatMessageDataObj {
  type: string;
  payload: FloydChatData;
}

type ChatMessageDataType = ChatMessageDataObj | string | undefined;

interface ChatMessageEvent extends MessageEvent {
  data: ChatMessageDataType;
}

interface Props {
  className?: string;
  name: string;
  isFakeChat?: boolean;
  onChatEnter?: (data: FloydChatData) => void;
  onChatExit?: (data: FloydChatData) => void;
  floydEnv: string;
}

class Chat extends React.Component<Props> {
  public static FLOYD_URLS = {
    test:
      'https://renderer-floyd-master.hamster.yandex.ru/chat/business?config=development&chatIframeOrigin=https://renderer-chat-dev.hamster.yandex.ru',
    prod: 'https://yandex.ru/chat/business',
  };

  public static DELAY_EXIT_EVENT = 100;

  public static defaultProps = {
    name: 'chat',
    isFakeChat: false,
    floydEnv: 'prod',
  };

  private iframeRef = React.createRef<HTMLIFrameElement>();

  private exitTimer: number;

  public componentDidMount() {
    window.addEventListener('message', this.handleMessage);
  }

  public componentWillUnmount() {
    window.removeEventListener('message', this.handleMessage);
  }

  private handleMessage = (e: ChatMessageEvent) => {
    let { data } = e;
    const { source } = e;

    /* process message only from chat */
    if (!(this.iframeRef.current && this.iframeRef.current.contentWindow === source)) {
      return;
    }

    if (!data) {
      return;
    }

    if (typeof data === 'object') {
      data = data as ChatMessageDataObj;

      if (!data.payload || !data.payload.chatId) {
        return;
      }
    }

    if (typeof data === 'string') {
      try {
        data = JSON.parse(data) as ChatMessageDataObj;
      } catch (error) {
        // eslint-disable-next-line no-console
        console.log('Fload json parse error: ', error);

        return;
      }
    }

    if (data.type === 'chat-enter') {
      this.handleChatEnter(data.payload);
    }

    if (data.type === 'chat-leave') {
      this.clearExitTimer();
      this.exitTimer = window.setTimeout(this.handleChatExit, Chat.DELAY_EXIT_EVENT, data.payload);
    }
  };

  private clearExitTimer = () => {
    window.clearTimeout(this.exitTimer);
  };

  private handleChatEnter = (data: FloydChatData) => {
    this.clearExitTimer();
    if (this.props.onChatEnter) {
      this.props.onChatEnter(data);
    }
  };

  private handleChatExit = (data: FloydChatData) => {
    this.clearExitTimer();
    if (this.props.onChatExit) {
      this.props.onChatExit(data);
    }
  };

  public render() {
    const { className, isFakeChat, floydEnv, ...other } = this.props;

    if (isFakeChat) {
      return <FakeChat className={className} />;
    }

    return (
      <iframe
        title="chat"
        src={Chat.FLOYD_URLS[floydEnv]}
        ref={this.iframeRef}
        {...other}
        className={cx(className, css.b)}
      />
    );
  }
}

const mapState = (state) => ({
  floydEnv: state && state.root && state.root.info && state.root.info.testMode ? 'test' : 'prod',
});

export default connect(mapState)(Chat);
