import React from 'react';
import debounce from 'lodash/debounce';
import cx from 'classnames';
import Categorization from 'modules/categorization';
import { Categorization as ICategorization } from 'modules/chat/types';
import css from './Layout.scss';

const DEFAULT_REQUIRE_MARKUP = 1;

interface Props {
  onTipClick: (url: string) => void;
  hidden?: boolean;
  value?: ICategorization;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  chat: any;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  onChange: (data: any) => void;
}

class ChatCategorization extends React.Component<Props> {
  private static DEBOUNCE_DELAY = 300;

  private lastValue: ICategorization | null;

  private debouncedChangeHandler = debounce((payload: ICategorization) => {
    this.props.onChange({
      categorization: payload,
      chat: this.props.chat,
    });
    this.lastValue = null;
  }, ChatCategorization.DEBOUNCE_DELAY);

  public constructor(props: Props) {
    super(props);
    this.lastValue = null;
  }

  public componentDidUpdate(prevProps) {
    const { lastValue } = this;
    if (
      !lastValue ||
      !prevProps.chat ||
      !prevProps.chat.chatId ||
      !this.props.chat ||
      !this.props.chat.chatId ||
      prevProps.chat.chatId === this.props.chat.chatId
    ) {
      return;
    }

    this.debouncedChangeHandler.cancel();
    this.props.onChange({
      categorization: lastValue,
      chat: prevProps.chat,
    });
    this.lastValue = null;
  }

  public componentWillUnmount() {
    const { lastValue } = this;
    if (!lastValue) {
      return;
    }

    this.debouncedChangeHandler.cancel();
    this.props.onChange({
      categorization: lastValue,
      chat: this.props.chat,
    });
    this.lastValue = null;
  }

  private handleChange = (payload: ICategorization) => {
    this.lastValue = { ...payload };
    this.debouncedChangeHandler(payload);
  };

  public render() {
    return (
      <Categorization
        className={cx(css.b__categorization, {
          [css.b__categorization_hidden]: this.props.hidden,
        })}
        onTipClick={this.props.onTipClick}
        value={this.props.value}
        onChange={this.handleChange}
        workplace="Chat"
        isRequireMarkup={DEFAULT_REQUIRE_MARKUP}
      />
    );
  }
}

export default ChatCategorization;
