import * as React from 'react';
import ClientSelectInputWithModal from 'modules/clientSelectInputWithModal';
import { FloydChatData, Categorization as ICategorazation, Tip as ITip } from 'modules/chat/types';
import { createPageLink, AppModuleString } from 'modules/requestForm2';
import Link from '@crm/components/dist/lego2/react-router/Link';
import Tip from 'components/Tip';
import Empty from 'components/Empty';
import { Account } from 'pages/accounts/modules/info';
import { Toolbar } from 'components/Layout';
import Panel from '../Panel';
import Categorization from './Categorization';
import Chat from '../Chat';
import css from './Layout.scss';

const messageEmptyAccount = (
  <div className={css.b__message}>
    <p>К этому чату ещё не привязан аккаунт.</p>
    <p>Найдите аккаунт по логину, и мы запомним его.</p>
  </div>
);

const messageEmptyChat = (
  <div className={css.b__message}>
    <p>Чат не выбран</p>
    <p className={css.b__messageSmall}>
      Если у вас не отображается чат, возможно, вы не вошли в паспорт на passport.yandex.ru
    </p>
  </div>
);

interface Account {
  id: number;
  name: string;
}

export interface Props {
  accountId?: number;
  chat?: {
    chatId: string;
    orgId: string;
  };
  onTipClick: (url: string) => void;
  tip?: ITip;
  categorization?: ICategorazation;
  onChangeAccount: (account: Account) => void;
  onCategorizationChange: (data: unknown) => void;
  onChatEnter: (data: FloydChatData) => void;
  onChatExit: (data: FloydChatData) => void;
}

const Layout = ({
  onChangeAccount,
  onCategorizationChange,
  onChatEnter,
  onChatExit,
  onTipClick,
  accountId,
  chat,
  categorization,
  tip,
}: Props) => {
  let accountBlock: React.ReactNode | null = null;
  const chatId = chat && chat.chatId;
  if (!chatId) {
    accountBlock = <Empty icon="question_answer" text={messageEmptyChat} />;
  } else if (!accountId) {
    accountBlock = <Empty icon="person" text={messageEmptyAccount} />;
  } else {
    accountBlock = (
      <>
        {tip && <Tip tip={tip} className={css.b__tip} />}
        <Account id={accountId} chatId={chatId} isCompact mode={Account.MODES.CHAT} />
      </>
    );
  }

  return (
    <div className={css.b}>
      <Toolbar className={css.b__toolbar}>
        <Toolbar.Left>
          <Toolbar.Title>
            <Toolbar.TitleName>Чаты</Toolbar.TitleName>
            <ClientSelectInputWithModal
              onChange={onChangeAccount}
              disabled={!chatId}
              canCreateLead
            />
            {accountId != null && (
              <Link
                to={createPageLink({
                  data: { accountId, byChat: chat },
                  context: { module: AppModuleString.Chat, action: 'Chat' },
                })}
                target="_blank"
              >
                Заявка
              </Link>
            )}
          </Toolbar.Title>
        </Toolbar.Left>
      </Toolbar>
      <div className={css.b__content}>
        <div className={css.b__account}>{accountBlock}</div>
        <div className={css.b__chatWrap}>
          <Chat className={css.b__chat} onChatEnter={onChatEnter} onChatExit={onChatExit} />
          <Categorization
            chat={chat}
            value={categorization}
            onChange={onCategorizationChange}
            onTipClick={onTipClick}
            hidden={accountId == null}
          />
        </div>
        {accountId != null ? (
          <Panel accountId={accountId} />
        ) : (
          <div className={css.b__historyEmpty} />
        )}
      </div>
    </div>
  );
};

export default Layout;
