import features from 'constants/features';
import { jsonApiCall } from 'api/common';
import { formatToBackend as saveAdapter } from 'modules/categorization';
import { actions as iframeActions } from 'components/Iframe';
import panelRedux from 'modules/panel/redux';
import {
  FloydChatData,
  BackendCategorization,
  Categorization,
  Account,
  State,
  Chat,
  Tip,
} from '../types';
import { PANEL_NAME, TIPS_PANEL_ID } from '../components/Panel';
import floydToCrmChatData from '../utils/floydToCrmChatData';
import { getChatData } from './selectors';
import * as actionTypes from './actionTypes';
import categorizationAdapter from '../utils/categorizationAdapter';

interface Action<P> {
  type: string;
  payload?: P;
}

type ChangeCategories = (payload: Categorization | null) => Action<Categorization | null>;
export const changeCategories: ChangeCategories = (payload) => ({
  type: actionTypes.CHANGE_CATEGORIES,
  payload,
});

type ChangeAccount = (account: Account | null) => Action<Account | null>;
export const changeAccount: ChangeAccount = (account) => ({
  type: actionTypes.CHANGE_ACCOUNT,
  payload: account,
});

type ChangeTip = (tip: Tip | null | undefined) => Action<Tip | null | undefined>;
export const changeTip: ChangeTip = (tip) => ({
  type: actionTypes.CHANGE_TIP,
  payload: tip,
});

type SaveCategories = (payload: {
  categorization: Categorization;
  chat: Chat;
}) => (
  dispatch,
  getState: () => {
    chat: State;
  },
) => Promise<{}>;
export const saveCategories: SaveCategories = (payload) => () => {
  return jsonApiCall({
    url: '/actions/chat/saveCategories',
    data: {
      chat: floydToCrmChatData(payload.chat),
      categories: saveAdapter(payload.categorization),
    },
  });
};

export const connectAccount = (account: Account) => (dispatch, getState) => {
  const chat = floydToCrmChatData(getChatData(getState()));
  return jsonApiCall({
    url: '/actions/chat/setAccount',
    data: { account, chat },
  })
    .then((data) => {
      dispatch(changeAccount(account));

      return data;
    })
    .then(() => {
      return jsonApiCall({
        url: '/view/chat/tip',
        data: { ...account, chat },
      });
    })
    .then(({ tip }: { tip: Tip }) => {
      dispatch(changeTip(tip));
    });
};

interface ChatEnterResult {
  account: Account;
  categorization?: BackendCategorization;
  tip?: Tip;
}
export const chatEnter = (data: FloydChatData) => (dispatch) => {
  dispatch({ type: actionTypes.CHAT_ENTER, payload: data });

  jsonApiCall({ url: '/view/chat/info', data: { chat: floydToCrmChatData(data) } }).then(
    (result: ChatEnterResult) => {
      if (result && result.account) {
        const categorization = categorizationAdapter(result.categorization);
        dispatch(changeAccount(result.account));
        dispatch(changeCategories(categorization));
        dispatch(changeTip(result.tip));
      } else {
        dispatch(changeAccount(null));
        dispatch(
          changeCategories({
            categories: {},
            selectedCategories: {},
          }),
        );
        dispatch(changeTip(null));
      }

      return result;
    },
  );
};

export const chatExit = (/* data: FloydChatData */) => (dispatch) => {
  dispatch({ type: actionTypes.CHAT_EXIT });
};

export const openTip = (url: string) => (dispatch) => {
  if (features.openTipInNewTab) {
    window.open(url);
  } else {
    dispatch(iframeActions.setSrc(url));
    dispatch(panelRedux.actions.setPanelId(PANEL_NAME, TIPS_PANEL_ID, false));
  }
};

export const destroy = () => ({
  type: actionTypes.DESTROY,
});
