import React from 'react';
import { connect } from 'react-redux';
import { reduxForm, Field, FieldArray } from 'redux-form';
import { cnTheme } from '@yandex-lego/components/Theme';
import { crmTheme } from '@crm/components/dist/lego2/Theme/presets/themes';
import cx from 'classnames';
import FieldTreeSelect from 'requestForms/components/FieldTreeSelect';
import Button from '@crm/components/dist/lego2/Button';
import FieldDateRange from 'requestForms/components/FieldDateRange';
import createI18N from '@yandex-int/i18n';
import FilterToggle from 'lego/components/FilterToggle';
import Icon from 'lego/components/Icon';
import TextInput from 'lego/redux-form/TextInput';
import reduxFormComponent from 'lego/redux-form';
import { apiMail as apiTags, TagsField as Tags } from 'modules/tags';
import * as keyset from './Filters.i18n';
import css from './Filters.scss';
import * as selectors from '../../selectors';
import { toggleWithFilters, toggleShowFilters } from '../../actions';
import { POPUP_OPEN_DURATION } from '../../constants';

const i18n = createI18N(keyset);
const i18nTicket = i18n('ticket');
const i18nOpportunity = i18n('opportunity');
const i18nRequest = i18n('request');
const i18nSharedCall = i18n('sharedCall');
const i18nPersonalCall = i18n('personalCall');
const i18nPersonalMail = i18n('personalMail');
const i18nComment = i18n('comment');
const i18nMassmail = i18n('massmail');
const i18nStTicket = i18n('stTicket');
const i18nChat = i18n('chat');
const i18nTask = i18n('task');
const i18nApproval = i18n('approval');
const i18nSearch = i18n('search');
const i18nFilters = i18n('filters');
const i18nApply = i18n('apply');
const i18nCategory = i18n('category');
const i18nAll = i18n('all');
const i18nStatus = i18n('status');
const i18nStatusOpen = i18n('statusOpen');
const i18nStatusInProgress = i18n('statusInProgress');
const i18nStatusClosed = i18n('statusClosed');
const i18nStatusDelayed = i18n('statusDelayed');
const i18nResolution = i18n('resolution');
const i18nResolved = i18n('resolved');
const i18nOnCall = i18n('onCall');
const i18nDup = i18n('dup');
const i18nSpam = i18n('spam');
const i18nPending = i18n('pending');
const i18nModeration = i18n('moderation');
const i18nNoReply = i18n('noReply');
const i18nRequiredInfo = i18n('requiredInfo');
const i18nIncorrect = i18n('incorrect');
const i18nMyCommunications = i18n('myCommunications');
const i18nLinkedAccounts = i18n('linkedAccounts');

const TYPES = [
  {
    value: 'ticket',
    icon: (iconCls) => <Icon className={iconCls} pack={Icon.PACKS.FONTAWESOME} icon="ticket" />,
    title: i18nTicket,
  },
  {
    value: 'request',
    icon: (iconCls) => <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="bookmark" />,
    title: i18nRequest,
  },
  {
    value: 'opportunity',
    icon: (iconCls) => <Icon className={iconCls} pack={Icon.PACKS.FONTAWESOME} icon="briefcase" />,
    title: i18nOpportunity,
  },
  {
    value: 'sharedCall',
    icon: (iconCls) => <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="phone" />,
    title: i18nSharedCall,
  },
  {
    value: 'personalCall',
    icon: (iconCls) => <Icon className={iconCls} type="personal-call" />,
    title: i18nPersonalCall,
  },
  {
    value: 'personalMail',
    icon: (iconCls) => <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="mail" />,
    title: i18nPersonalMail,
  },
  {
    value: 'accountComment',
    icon: (iconCls) => <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="note" />,
    title: i18nComment,
  },
  {
    value: 'massmail',
    icon: (iconCls) => <Icon className={iconCls} type="svg" icon="massmail" />,
    title: i18nMassmail,
  },
  {
    value: 'accountSt',
    icon: (iconCls) => <Icon className={iconCls} type="svg" icon="startrek" mode="grey" />,
    title: i18nStTicket,
  },
  {
    value: 'chatChunk',
    icon: (iconCls) => (
      <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="question_answer" />
    ),
    title: i18nChat,
  },
  {
    value: 'task',
    icon: (iconCls) => <Icon className={iconCls} type="svg" icon="issues" />,
    title: i18nTask,
  },
  {
    value: 'approval',
    icon: (iconCls) => <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="done_all" />,
    title: i18nApproval,
  },
];

interface Props {
  withFilters: boolean;
  showFilters: boolean;
  toggleWithFilters: () => {};
  toggleShowFilters: () => {};
  handleSubmit: () => {};
}

const Filters = ({
  withFilters,
  showFilters,
  toggleWithFilters, // eslint-disable-line no-shadow
  toggleShowFilters, // eslint-disable-line no-shadow
  handleSubmit,
}: Props) => (
  <form className={cx(css.b, cnTheme({ root: crmTheme.root }))} onSubmit={handleSubmit}>
    <div className={css.b__row}>
      <div className={cx(css.b__cell, css.b__cell_max)}>
        <Field
          placeholder={i18nSearch}
          size="xs"
          name="text"
          component={TextInput}
          autoFocus={POPUP_OPEN_DURATION}
        />
      </div>
      <div className={css.b__cell}>
        <Button
          disabled={!withFilters}
          onClick={toggleShowFilters}
          pin="round-clear"
          size="xs"
          iconRight={(iconCls) => (
            <Icon className={iconCls} type="arrow" direction={showFilters ? 'top' : 'bottom'} />
          )}
        >
          {i18nFilters}
        </Button>
        <FilterToggle
          onChange={toggleWithFilters}
          value={withFilters}
          pin="brick-round"
          size="xs"
        />
      </div>
      <div className={css.b__cell}>
        <Button type="submit" view="action" size="xs">
          {i18nApply}
        </Button>
      </div>
    </div>
    {withFilters && showFilters && (
      <div className={cx(css.b__row, css.b__row_column)}>
        <div className={css.b__row}>
          <Field
            name="types"
            component={reduxFormComponent.ButtonGroup}
            size="xs"
            mode="check"
            width="max"
            items={TYPES}
            fontSize="l"
          />
        </div>
        <div className={css.b__row}>
          <Field
            labelPrefix={i18nCategory}
            labelWidth={100}
            component={FieldTreeSelect}
            zIndexGroupLevel={4}
            name="categoryIds"
            size="xs"
            mode="hierarchy-check"
            endpoint="/view/support/category/list/all"
            defaultValueText={i18nAll}
            props={{ name: 'categoryId' }}
          />
        </div>
        <div className={css.b__row}>
          <div className={cx(css.b__cell, css.b__cell_max)}>
            <Field
              name="stateIds"
              component={reduxFormComponent.Select}
              size="xs"
              placeholder={i18nStatus}
              width="max"
              zIndexGroupLevel={4}
              hasEmptyValue={false}
              options={[
                { value: '2', content: i18nStatusOpen },
                { value: '3', content: i18nStatusInProgress },
                { value: '4', content: i18nStatusClosed },
                { value: '5', content: i18nStatusDelayed },
              ]}
            />
          </div>
          <div className={cx(css.b__cell, css.b__cell_max)}>
            <Field
              name="resolutions"
              component={reduxFormComponent.Select}
              size="xs"
              placeholder={i18nResolution}
              width="max"
              zIndexGroupLevel={4}
              hasEmptyValue={false}
              options={[
                { value: 'resolved', content: i18nResolved },
                { value: 'oncall', content: i18nOnCall },
                { value: 'duplicate', content: i18nDup },
                { value: 'spam', content: i18nSpam },
                { value: 'pendingAdjacenter', content: i18nPending },
                { value: 'moderation', content: i18nModeration },
                { value: 'noreply', content: i18nNoReply },
                { value: 'requiredInfo', content: i18nRequiredInfo },
                { value: 'uncorrect', content: i18nIncorrect },
              ]}
            />
          </div>
        </div>
        <div className={css.b__row}>
          <div className={css.b__cell}>
            <Field
              name="myCommunications"
              component={reduxFormComponent.CheckBox}
              lineHeightSize="xs"
            >
              {i18nMyCommunications}
            </Field>
          </div>
          <div className={css.b__cell}>
            <Field
              name="linkedAccounts"
              component={reduxFormComponent.CheckBox}
              lineHeightSize="xs"
            >
              {i18nLinkedAccounts}
            </Field>
          </div>
          <div className={cx(css.b__cell, css.b__cell_date, css.b__cell_last)}>
            <FieldDateRange startName="from" endName="to" size="xs" popupPosition="right" />
          </div>
        </div>
        <div className={css.b__row}>
          <div className={cx(css.b__cell, css.b__cell_max)}>
            <FieldArray
              name="tagIds"
              component={Tags}
              props={{
                getTags: () => apiTags.get(),
              }}
            />
          </div>
        </div>
      </div>
    )}
  </form>
);

export default connect(
  (state) => ({
    withFilters: selectors.withFilters(state),
    showFilters: selectors.showFilters(state),
    initialValues: selectors.filter(state),
  }),
  { toggleWithFilters, toggleShowFilters },
)(
  reduxForm<{}, Props>({
    form: 'clientFilters',
    initialValues: {
      stateIds: [],
      resolutions: [],
    },
  })(Filters),
);
