import React from 'react';
import { Issue } from 'modules/issues';
import { IssueType } from 'modules/issues/types';
import createLink from 'modules/issues/utils/createLink';
import createLinkByType from 'modules/issues/utils/createLinkByType';
import Link from '@crm/components/dist/lego2/Link';
import sp from 'utils/stopPropagation';
import createI18N from '@yandex-int/i18n';
import Name from 'components/Name';
import IssuesIcon from 'lego/components/Icon/svg/Issues';
import { ISSUE_TYPES } from 'constants/backendTypes';
import * as keyset from '../HistoryItems.i18n';
import ItemDate from '../ItemDate';
import ItemColumnDate from '../ItemColumnDate';
import ItemColumn from '../ItemColumn';
import ItemWrap from '../ItemWrap';
import ItemHeader from '../ItemHeader';
import ItemIcon from '../ItemIcon';

const i18n = createI18N(keyset);

const ISSUE_ICON = {
  [ISSUE_TYPES.TASK]: (
    <ItemIcon type="Request">
      <IssuesIcon />
    </ItemIcon>
  ),
  [ISSUE_TYPES.APPROVAL]: <ItemIcon type={ISSUE_TYPES.APPROVAL} />,
  [ISSUE_TYPES.TICKET]: <ItemIcon type={ISSUE_TYPES.TICKET} />,
  [ISSUE_TYPES.OPPORTUNITY]: <ItemIcon type={ISSUE_TYPES.OPPORTUNITY} />,
};

interface Dict {
  id: number;
  name: string;
}

interface Data {
  id: number;
  /** Translated type */
  /* todo: remove optional when backend support CRM-9625 */
  type?: string;
  /* todo: remove optional when backend support CRM-9625 */
  typeId?: IssueType;
  state: Dict;
  name: string;
  date: string;
  // eslint-disable-next-line
  owner: any;
}

interface IssueHistoryProps {
  data: Data;
  toggle?: boolean;
  onToggle: () => void;
  // eslint-disable-next-line
  item: any;
  /* todo: change when backend support CRM-9625 to 'Issue' */
  type: 'Ticket' | 'Task' | 'Approval';
}

const IssueHistory: React.FC<IssueHistoryProps> = (props) => {
  const { toggle, onToggle, data, item, type } = props;
  const { id, name, date, owner, state, type: issueType, typeId } = data;

  let iconType = type;

  let link = '';
  if (typeof typeId === 'number') {
    link = createLink({ typeId, id });
    iconType = IssueType[typeId] as typeof type;
  } else {
    link = createLinkByType({ type, id });
  }

  let title = '';
  if (issueType) {
    title = `${issueType} ${id}`;
  } else {
    /* todo: remove after CRM-9625 from tanker */
    title = i18n('issue', { id });
  }

  return (
    <ItemWrap isOpen={toggle}>
      <ItemHeader onClick={onToggle}>
        {ISSUE_ICON[iconType]}
        <ItemColumn title={title} subTitle={name} />
        <ItemColumn
          title={owner ? <Name name={owner.name} /> : null}
          subTitle={state && state.name}
        />
        <ItemColumnDate item={item}>
          <Link href={link} target="_blank" onClick={sp}>
            <ItemDate date={date} />
          </Link>
        </ItemColumnDate>
      </ItemHeader>
      {toggle && (
        <div>
          <Issue
            id={id}
            showAttributes={false}
            showExpandWidgets={false}
            maxAccess={1}
            timelineHeight={500}
          />
        </div>
      )}
    </ItemWrap>
  );
};

export default IssueHistory;
