import React from 'react';
import PropTypes from 'prop-types';
import LinkModule from 'components/LinkModule';
import Name from 'components/Name';
import createI18N from '@yandex-int/i18n';
import * as keyset from '../HistoryItems.i18n';
import ItemDate from '../ItemDate';
import ItemColumnDate from '../ItemColumnDate';
import ItemColumn from '../ItemColumn';
import ItemWrap from '../ItemWrap';
import ItemHeader from '../ItemHeader';
import ItemIcon from '../ItemIcon';
import css from './styles/styles';

const i18n = createI18N(keyset);
const i18nCallTitle = i18n('callTitle');
const i18nTicketTitle = i18n('ticketTitle');
const i18nRequestTitle = i18n('requestTitle');
const i18nDupTitle = i18n('dupTitle');
const i18nCurrentTitle = i18n('currentTitle');

const TITLE_MAP = {
  Call: i18nCallTitle,
  Ticket: i18nTicketTitle,
  Request: i18nRequestTitle,
};

const PROPS_MAP = {
  Call: 'call',
  Ticket: 'ticket',
  Request: 'request',
};

export default class Item extends React.Component {
  static propTypes = {
    id: PropTypes.number,
    type: PropTypes.string,
    show: PropTypes.bool,
    connection: PropTypes.string,
    children: PropTypes.node,
    onClick: PropTypes.func.isRequired,
    item: PropTypes.instanceOf(Object),
  };

  static defaultProps = {
    id: undefined,
    type: undefined,
    show: undefined,
    connection: undefined,
    children: undefined,
    item: undefined,
  };

  static getTitle(data, type) {
    let title = TITLE_MAP[type];
    const { number } = data;
    if (number) {
      title += ` ${data.number}`;
    }

    return title;
  }

  static getState(data) {
    if (!data.state) {
      return null;
    }

    const resolution = data.resolution ? ` (${data.resolution.name})` : '';

    return data.state.name + resolution;
  }

  render() {
    const { id, type, show, connection, children, item } = this.props;

    if (!type) {
      return null;
    }

    const data = this.props[PROPS_MAP[type]];

    let connectionClass = css.header.connection;
    let title;
    switch (connection) {
      case 'Duplicate':
        title = i18nDupTitle;
        connectionClass = css.header.connection_duplicate;
        break;
      case 'Current':
        title = i18nCurrentTitle;
        connectionClass = css.header.connection_current;
        break;
      default:
        break;
    }

    return (
      <ItemWrap isOpen={show}>
        <ItemHeader onClick={this.props.onClick} title={title}>
          <div className={connectionClass} />
          <ItemIcon type={type} />
          <ItemColumn title={Item.getTitle(data, type)} subTitle={data.category} />
          <ItemColumn
            title={data.owner ? <Name name={data.owner.name} /> : null}
            subTitle={Item.getState(data)}
          />
          <ItemColumnDate item={item}>
            <LinkModule type={type} id={id} isStopPropagation>
              <ItemDate date={data.date} />
            </LinkModule>
          </ItemColumnDate>
        </ItemHeader>
        {show && <div className={css.item.content}>{children}</div>}
      </ItemWrap>
    );
  }
}
