import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { actions as listActions } from 'containers/InfiniteList';
import { NULL_CLIENT } from 'constants/index';
import isModifiedEvent from 'utils/isModifiedEvent';
import { localstorageKey, lsAccountTypeKey } from './constants';
import Form from './Form';
import { loadClient } from './api';

const itemKey = item => `${item.id}_${item.searchKey}`;

class FormController extends React.Component {
  constructor(props) {
    super(props);

    const { forceAccountTypes, forceSearchKeys } = props;

    // Инициализируем поле searchKey
    let defaultSearchKey;
    if (forceSearchKeys) {
      defaultSearchKey = forceSearchKeys;
    } else {
      const localstorageKeyStr = localStorage.getItem(localstorageKey);
      let localStorageSearchKey;

      try {
        localStorageSearchKey = JSON.parse(localstorageKeyStr);
      } catch (e) {
        /* no catch */
      }

      if (Array.isArray(localStorageSearchKey) && localStorageSearchKey.length) {
        defaultSearchKey = localStorageSearchKey;
      }

      if (!defaultSearchKey && localstorageKeyStr) {
        defaultSearchKey = [localstorageKeyStr];
      }

      if (!defaultSearchKey) {
        defaultSearchKey = [props.defaultSearchKey];
      }
    }

    // Инициализируем поле accountType
    this.componentDefaultAccountType = 'any';

    let arrAccountTypes;
    if (forceAccountTypes) {
      arrAccountTypes = forceAccountTypes;
    } else {
      arrAccountTypes = [this.componentDefaultAccountType];
      try {
        const localStorageAccountTypes = JSON.parse(localStorage.getItem(lsAccountTypeKey));
        if (Array.isArray(localStorageAccountTypes) && localStorageAccountTypes.length) {
          arrAccountTypes = localStorageAccountTypes;
        }
      } catch (e) {
        /* no catch */
      }
    }

    this.state = {
      searchKey: defaultSearchKey,
      accountType: arrAccountTypes,
      searchValue: props.defaultSearchValue,
      currentItem: null,
    };
  }

  componentDidMount() {
    if (this.state.searchValue) {
      this.onSearchClick();
    }
  }

  onChangeSearchText = value => {
    this.setState({
      searchValue: value,
    });
  };

  onChangeSearchKey = value => {
    try {
      localStorage.setItem(localstorageKey, JSON.stringify(value));
    } catch (e) {
      /* no catch */
    }

    this.setState({
      searchKey: value,
    });
  };

  onChangeAccountType = value => {
    let updatedList;

    if (
      this.state.accountType.indexOf(this.componentDefaultAccountType) === -1 &&
      value.length > 1 &&
      value.indexOf(this.componentDefaultAccountType) !== -1
    ) {
      updatedList = [this.componentDefaultAccountType];
    } else if (value.length !== 1 && value.indexOf(this.componentDefaultAccountType) !== -1) {
      updatedList = value.filter(v => v !== this.componentDefaultAccountType);
    } else {
      updatedList = value;
    }

    try {
      localStorage.setItem(lsAccountTypeKey, JSON.stringify(updatedList));
    } catch (e) {
      /* no catch */
    }

    this.setState({
      accountType: updatedList,
    });
  };

  onSelectClientClick = (id, item, event) => {
    if (!isModifiedEvent(event)) {
      event.preventDefault();

      this.setState({
        currentItem: item,
      });
    }
  };

  onSelectClientChange = (id, props) => {
    this.setState({
      currentItem: props ? props.item : null,
    });
  };

  onSearchClick = () => {
    this.props.dispatch(listActions.reset('clientSearch'));
    this.props.dispatch(listActions.load('clientSearch'));
    this.setState({ currentItem: null });
  };

  onNewClient = value => {
    if (value) {
      this.setState({
        currentItem: NULL_CLIENT,
      });
    } else {
      this.setState({
        currentItem: null,
      });
    }
  };

  onSubmit = () => {
    if (this.state.currentItem) {
      const submitValue = this.props.onSubmit(this.state.currentItem);
      const { onSubmitSuccess } = this.props;
      if (typeof onSubmitSuccess === 'function') {
        if (submitValue && typeof submitValue.then === 'function') {
          submitValue.then(data => onSubmitSuccess(data));
        } else {
          onSubmitSuccess();
        }
      }
    }
  };

  render() {
    const { forceAccountTypes, forceSearchKeys, ...passPropsThrough } = this.props;

    return (
      <Form
        {...passPropsThrough}
        onSubmit={this.onSubmit}
        onNewClient={this.onNewClient}
        isNewClient={this.state.currentItem && this.state.currentItem.id === NULL_CLIENT.id}
        searchText={this.state.searchValue}
        searchKey={this.state.searchKey}
        accountType={this.state.accountType}
        onChangeSearchText={this.onChangeSearchText}
        onChangeSearchKey={this.onChangeSearchKey}
        onChangeAccountType={this.onChangeAccountType}
        disabledAccountType={!!forceAccountTypes}
        disabledKeys={!!forceSearchKeys}
        onSelectClientClick={this.onSelectClientClick}
        onSelectClientChange={this.onSelectClientChange}
        selectedClientId={this.state.currentItem && itemKey(this.state.currentItem)}
        onSearchClick={this.onSearchClick}
        itemKey={itemKey}
        onLoad={pagination =>
          loadClient({
            ...(Array.isArray(this.state.searchKey) ? this.state.searchKey : []).reduce((ac, c) => {
              // eslint-disable-next-line no-param-reassign
              ac[c] = this.state.searchValue;
              return ac;
            }, {}),
            accountType: this.state.accountType,
            ...pagination,
          })
        }
        valid={!!this.state.currentItem}
      />
    );
  }
}

FormController.propTypes = {
  defaultSearchKey: PropTypes.string,
  defaultSearchValue: PropTypes.string,
  defaultAccountType: PropTypes.string,
  forceAccountTypes: PropTypes.arrayOf(PropTypes.string),
  forceSearchKeys: PropTypes.arrayOf(PropTypes.string),
  onSubmitSuccess: PropTypes.func.isRequired,
  onSubmit: PropTypes.func.isRequired,
  dispatch: PropTypes.func.isRequired,
};

FormController.defaultProps = {
  defaultSearchKey: 'campaignId',
  defaultAccountType: undefined,
  forceAccountTypes: undefined,
  forceSearchKeys: undefined,
  defaultSearchValue: '',
};

export default connect()(FormController);
