import { createElement, Component } from 'react';
import PropTypes from 'prop-types';
import enhanceWithClickOutside from 'react-click-outside';
import LocalStorage from 'utils/localStorage';
import ClientSelectInput from './ClientSelectInput';
import config from './config';
import { load } from './api';

const LS_SEARCH_KEYS = 'clientSelectInput.keys';

class ClientSelectInputContainer extends Component {
  static propTypes = {
    onChange: PropTypes.func,
    forceSearchKeys: PropTypes.instanceOf(Array),
  };

  static defaultProps = {
    onChange: undefined,
    forceSearchKeys: undefined,
  };

  constructor(props) {
    super(props);

    const { forceSearchKeys } = props;

    let keys = forceSearchKeys || LocalStorage.getItem(LS_SEARCH_KEYS);
    if (!Array.isArray(keys)) {
      keys = [config.resetItem];
    }

    this.state = {
      value: '',
      keys,
      isSearch: false,
      account: null,
    };
  }

  setAccount(account) {
    const { onChange } = this.props;
    if (typeof onChange === 'function') {
      onChange(account);
    }

    this.setState({
      account: null,
      isSearch: false,
    });
  }

  handleKeyComponent = e => {
    const { account } = this.state;
    if (e.key === 'Enter' && account) {
      this.setAccount(account);
    }
  };

  handleChangeValue = value => {
    this.setState({ value });
  };

  handleChangeKeys = value => {
    LocalStorage.setItem(LS_SEARCH_KEYS, value);
    this.setState({ keys: value });
  };

  handleKeyValue = e => {
    e.stopPropagation();
    if (e.key === 'Enter' && this.isValid()) {
      this.handleSearch();
    }
  };

  handleClickOutside = () => {
    if (this.state.isSearch) {
      this.setState({
        isSearch: false,
        account: null,
      });
    }
  };

  handleChangeAccount = (id, props) => {
    if (props) {
      this.setState({
        account: props.data,
      });
    }
  };

  handleSelectAccount = account => {
    this.setAccount(account);
  };

  isValid() {
    const { value, keys } = this.state;

    return value && Array.isArray(keys) && keys.length > 0;
  }

  handleSearch = () => {
    const trimValue = this.state.value.trim();

    this.setState({
      value: trimValue,
      isSearch: true,
      provider: args =>
        load({
          ...args,
          ...this.state.keys.reduce((pv, cv) => ({ ...pv, [cv]: trimValue }), {}),
        }),
    });
  };

  render() {
    const { account, ...passStateThrough } = this.state;
    const { forceSearchKeys, ...other } = this.props;

    return createElement(ClientSelectInput, {
      ...passStateThrough,
      ...other,
      accountId: (account || {}).id,
      onChangeValue: this.handleChangeValue,
      onChangeKeys: this.handleChangeKeys,
      onChangeAccount: this.handleChangeAccount,
      onSelectAccount: this.handleSelectAccount,
      onKeyComponent: this.handleKeyComponent,
      onKeyValue: this.handleKeyValue,
      onSearch: this.handleSearch,
      disabledKeys: !!forceSearchKeys,
      disabledFind: !this.isValid(),
    });
  }
}

export default enhanceWithClickOutside(ClientSelectInputContainer);
