import * as React from 'react';
import Button from '@crm/components/dist/lego2/Button';
import { LegoSizeProp } from '@crm/components/dist/lego2/types';
import cx from 'classnames';
import { ModalForm as ClientFind } from 'modules/clientSelectForm';
import { ModalForm as ClientCreate, CreateLeadButton } from 'modules/clientCreateForm';
import ClientSelectInput from 'modules/clientSelectInput';
import css from './Input.scss';

const NULL_ACCOUNT = {
  id: 1,
  name: 'НУЛЕВОЙ КЛИЕНТ',
};

interface Account {
  id: number;
  name: string;
}

interface Props {
  onChange: (account: Account) => unknown;
  autoFocus?: boolean;
  disabledSetNewAccount?: boolean;
  disabled?: boolean;
  disabledCreateAccount?: boolean;
  disabledPinInput?: boolean;
  className?: string;
  size?: LegoSizeProp;
  forceSearchKeys?: string[];
  isNullAccount?: boolean;
  isCreateAccount?: boolean;
  hasInlineSearch?: boolean;
  hasModalSearch?: boolean;
  canCreateLead?: boolean;
  canShowPin?: boolean;
  multiline?: boolean;
  onPinButtonClick?: () => void;
}

const Input = (props: Props): React.ReactElement => {
  const {
    onChange,
    autoFocus,
    disabledSetNewAccount,
    className,
    size = 's',
    disabled,
    forceSearchKeys,
    isNullAccount = true,
    isCreateAccount,
    hasInlineSearch = true,
    hasModalSearch = true,
    multiline,
    canCreateLead,
    onPinButtonClick,
    canShowPin = true,
    disabledCreateAccount = false,
    disabledPinInput = false,
  } = props;

  const handleNullAccountCreate = React.useCallback(() => {
    onChange(NULL_ACCOUNT);
  }, [onChange]);

  return (
    <span
      className={cx(className, css.b, {
        [css.b_multiline]: multiline,
        [css.b_default]: !multiline,
      })}
    >
      {hasInlineSearch && (
        <ClientSelectInput
          onChange={onChange}
          autoFocus={autoFocus}
          disabled={disabled}
          size={size}
          forceSearchKeys={forceSearchKeys}
        />
      )}
      {onPinButtonClick && canShowPin && (
        <Button size={size} onClick={onPinButtonClick} disabled={disabledPinInput}>
          Ввести ПИН-код
        </Button>
      )}
      {isNullAccount && (
        <Button
          size={size}
          disabled={disabled || disabledSetNewAccount}
          onClick={handleNullAccountCreate}
        >
          Нулевой клиент
        </Button>
      )}
      {hasModalSearch && (
        <ClientFind
          name="SEARCH_CLIENT"
          onSubmit={onChange}
          forceSearchKeys={forceSearchKeys}
          newClient={isNullAccount}
        >
          {(modal, open) => (
            <Button onClick={open} size={size} disabled={disabled}>
              Выбрать аккаунт
            </Button>
          )}
        </ClientFind>
      )}
      {isCreateAccount && (
        <ClientCreate name="CREATE_CLIENT" onSubmit={onChange}>
          {(modal, open) => (
            <Button size={size} onClick={open} disabled={disabledCreateAccount}>
              Создать аккаунт
            </Button>
          )}
        </ClientCreate>
      )}
      {canCreateLead && <CreateLeadButton size={size} onCreate={onChange} disabled={disabled} />}
    </span>
  );
};

export default Input;
