import React from 'react';
import PropTypes from 'prop-types';
import get from 'lodash/get';
import { stateSaver } from 'modules/localStorage';
import saveUI from './saveUI';

const getFilterObject = (filters = {}, activeFilter) =>
  Object.values(filters).reduce((totalFilter, filter) => {
    if (filter && (filter.key === activeFilter || filter.isActive)) {
      let filterValue = filter.value;
      if (typeof filterValue !== 'object') {
        filterValue = { [filter.key]: filterValue };
      }

      return {
        ...totalFilter,
        ...filterValue,
      };
    }

    return totalFilter;
  }, {});

class Filter extends React.Component {
  static sharedProps = {};

  constructor(props) {
    super(props);

    const { sharedProps, name, isStore } = props;
    if (sharedProps) {
      this.setSharedProps(sharedProps);
    }

    if (isStore) {
      this.saveUI = saveUI(name);
      stateSaver.addHandler(this.saveUI);
    }

    this.state = {
      provider: props.provider,
    };
  }

  componentWillReceiveProps(nextProps) {
    const { activeFilter, filters } = nextProps;

    if (activeFilter !== this.props.activeFilter || filters !== this.props.filters) {
      const filter = getFilterObject(filters, activeFilter);

      this.setState({
        provider: args => this.props.provider({ ...args, ...filter }),
      });
    }
  }

  componentWillUnmount() {
    if (this.saveUI) {
      stateSaver.removeHandler(this.saveUI);
    }
  }

  setSharedProps = obj => {
    Filter.sharedProps[this.props.name] = obj;
  };

  getSharedProps = () => get(Filter.sharedProps, this.props.name);

  render() {
    return this.props.children({
      provider: this.state.provider,
      filter: this.props.filters[this.props.activeFilter] || {},
      getSharedProps: this.getSharedProps,
    });
  }
}

Filter.propTypes = {
  name: PropTypes.string.isRequired,
  sharedProps: PropTypes.instanceOf(Object),
  provider: PropTypes.func.isRequired,
  children: PropTypes.func.isRequired,
  activeFilter: PropTypes.string,
  isStore: PropTypes.bool,
  filters: PropTypes.instanceOf(Object),
};

Filter.defaultProps = {
  activeFilter: undefined,
  filters: {},
  isStore: false,
  sharedProps: {},
};

export default Filter;
