/* eslint-disable @typescript-eslint/no-explicit-any */
import React from 'react';
import { EventTarget } from 'event-target-shim';
import { Provider } from 'react-redux';
import store from 'store';
import { WebphoneOutgoingEventKind } from 'modules/webphone/services/WebphoneEventManager';
import { render, act, screen } from '@testing-library/react';
import 'modules/webphone/components/CallTo';
import { CallDirection, CallStatus } from 'modules/issues/redux/api/callApi';
import { ReduxContext } from '../../redux';
import { issueServicesContext } from '../IssueServicesContext';
import { IssueIdContext } from '../Issue/IssueIdContext';
import { CallTo } from './CallTo';

const mockRenderCallToProps = jest.fn();

jest.mock('modules/webphone/components/CallTo', () => {
  return {
    CallTo: (props) => {
      mockRenderCallToProps(props);

      return <button>call</button>;
    },
  };
});

const mockSaveCallLastId = { value: 99 };
const mockSaveCall = jest.fn((data) => () =>
  Promise.resolve({ data: { id: data.id ? data.id : ++mockSaveCallLastId.value } }),
);

const webphoneMock = new EventTarget();

const createCallToTestElement = ({ mockSaveCall, issueId = 1 }) => {
  return (
    <Provider store={store}>
      <IssueIdContext.Provider value={issueId}>
        <issueServicesContext.Provider
          value={{
            callAvailableService: {
              canCall() {
                return Promise.resolve({ canCall: true });
              },
            },
            callMetaService: {
              getCallMeta() {
                return Promise.resolve({
                  caller: 'caller',
                  associatedUnitId: 'associatedUnitId',
                });
              },
            },
          }}
        >
          <ReduxContext.Provider
            value={
              { slices: { timelineSlice: { asyncActions: { saveCall: mockSaveCall } } } } as any
            }
          >
            <CallTo
              phone="+79091111111"
              cardData={{ accountKikId: 1 }}
              phoneData={{ id: 2, phone: '+79091111111', phoneE164: '+79091111111' }}
            />
          </ReduxContext.Provider>
        </issueServicesContext.Provider>
      </IssueIdContext.Provider>
    </Provider>
  );
};

describe('Issue', () => {
  describe('CallTo', () => {
    beforeEach(() => {
      mockRenderCallToProps.mockClear();
      mockSaveCall.mockClear();
      mockSaveCallLastId.value = 99;
    });

    test('simple once call case', async () => {
      render(createCallToTestElement({ mockSaveCall }));

      mockRenderCallToProps.mock.calls[0][0].onCallInit({
        webphone: webphoneMock,
        meta: { associatedUnitId: 'associatedUnitId', caller: 'caller' },
      });

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.NewOutgoingCall,
          callId: 'callId',
        });
      });

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.CallEnd,
          callId: 'callId',
        });
      });

      expect(mockSaveCall).toBeCalledTimes(2);

      expect(mockSaveCall.mock.calls[0][0]).toStrictEqual({
        callId: 'callId',
        direction: CallDirection.Out,
        kikId: 2,
        status: CallStatus.InProgress,
        operatorPhoneNumber: 'caller',
        issueId: 1,
      });

      expect(mockSaveCall.mock.calls[1][0]).toStrictEqual({
        id: 100,
        callId: 'callId',
        direction: CallDirection.Out,
        kikId: 2,
        status: CallStatus.End,
        operatorPhoneNumber: 'caller',
        issueId: 1,
      });
    });

    test('second call', async () => {
      render(createCallToTestElement({ mockSaveCall }));

      mockRenderCallToProps.mock.calls[0][0].onCallInit({
        webphone: webphoneMock,
        meta: { associatedUnitId: 'associatedUnitId', caller: 'caller' },
      });

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.NewOutgoingCall,
          callId: 'callId',
        });
      });

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.CallEnd,
          callId: 'callId',
        });
      });

      mockRenderCallToProps.mock.calls[0][0].onCallInit({
        webphone: webphoneMock,
        meta: { associatedUnitId: 'associatedUnitId2', caller: 'caller2' },
      });

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.NewOutgoingCall,
          callId: 'callId2',
        });
      });

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.CallEnd,
          callId: 'callId2',
        });
      });

      expect(mockSaveCall).toBeCalledTimes(4);

      expect(mockSaveCall.mock.calls[2][0]).toStrictEqual({
        callId: 'callId2',
        direction: CallDirection.Out,
        kikId: 2,
        status: CallStatus.InProgress,
        operatorPhoneNumber: 'caller2',
        issueId: 1,
      });

      expect(mockSaveCall.mock.calls[3][0]).toStrictEqual({
        id: 101,
        callId: 'callId2',
        direction: CallDirection.Out,
        kikId: 2,
        status: CallStatus.End,
        operatorPhoneNumber: 'caller2',
        issueId: 1,
      });
    });

    it('should correct end call after rerender without component', async () => {
      const { rerender } = render(createCallToTestElement({ mockSaveCall }));

      mockRenderCallToProps.mock.calls[0][0].onCallInit({
        webphone: webphoneMock,
        meta: { associatedUnitId: 'associatedUnitId', caller: 'caller' },
      });

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.NewOutgoingCall,
          callId: 'callId',
        });
      });

      act(() => {
        rerender(<div>empty</div>);
      });

      expect(screen.queryByText('call')).not.toBeInTheDocument();
      expect(screen.getByText('empty')).toBeInTheDocument();

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.CallEnd,
          callId: 'callId',
        });
      });

      expect(mockSaveCall).toBeCalledTimes(2);

      expect(mockSaveCall.mock.calls[0][0]).toStrictEqual({
        callId: 'callId',
        direction: CallDirection.Out,
        kikId: 2,
        status: CallStatus.InProgress,
        operatorPhoneNumber: 'caller',
        issueId: 1,
      });

      expect(mockSaveCall.mock.calls[1][0]).toStrictEqual({
        id: 100,
        callId: 'callId',
        direction: CallDirection.Out,
        kikId: 2,
        status: CallStatus.End,
        operatorPhoneNumber: 'caller',
        issueId: 1,
      });
    });

    it('should correct end call after rerender with different issue', async () => {
      const { rerender } = render(createCallToTestElement({ mockSaveCall }));

      mockRenderCallToProps.mock.calls[0][0].onCallInit({
        webphone: webphoneMock,
        meta: { associatedUnitId: 'associatedUnitId', caller: 'caller' },
      });

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.NewOutgoingCall,
          callId: 'callId',
        });
      });

      act(() => {
        rerender(createCallToTestElement({ mockSaveCall, issueId: 2 }));
      });

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.CallEnd,
          callId: 'callId',
        });
      });

      expect(mockSaveCall).toBeCalledTimes(2);

      expect(mockSaveCall.mock.calls[0][0]).toStrictEqual({
        callId: 'callId',
        direction: CallDirection.Out,
        kikId: 2,
        status: CallStatus.InProgress,
        operatorPhoneNumber: 'caller',
        issueId: 1,
      });

      expect(mockSaveCall.mock.calls[1][0]).toStrictEqual({
        id: 100,
        callId: 'callId',
        direction: CallDirection.Out,
        kikId: 2,
        status: CallStatus.End,
        operatorPhoneNumber: 'caller',
        issueId: 1,
      });
    });

    it('should correct destroy with no error', async () => {
      render(createCallToTestElement({ mockSaveCall }));

      mockRenderCallToProps.mock.calls[0][0].onCallError(new Error('error'));
    });

    it('should correct destroy after new outgoing call', async () => {
      render(createCallToTestElement({ mockSaveCall }));

      mockRenderCallToProps.mock.calls[0][0].onCallInit({
        webphone: webphoneMock,
        meta: { associatedUnitId: 'associatedUnitId', caller: 'caller' },
      });

      mockRenderCallToProps.mock.calls[0][0].onCallError(new Error('error'));

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.NewOutgoingCall,
          callId: 'callId',
        });
      });

      await act(async () => {
        webphoneMock.dispatchEvent({
          type: WebphoneOutgoingEventKind.CallEnd,
          callId: 'callId',
        });
      });

      expect(mockSaveCall).toBeCalledTimes(0);
    });
  });
});
