import React, { FC, useCallback, useContext, useRef } from 'react';
import { useDispatch } from 'react-redux';
import { CallTo as BaseCallTo, OnCallInit } from 'modules/webphone/components/CallTo';
import { CallToProps } from 'components/Contact/Phone';
import {
  WebphoneEventManager,
  WebphoneOutgoingEventKind,
} from 'modules/webphone/services/WebphoneEventManager';
import { webphoneStateManager } from 'modules/webphone/services/WebphoneStateManager';
import { establishedCallEventListener } from 'modules/webphone/utils/establishedCallEventListener';
import { CallDirection, CallStatus } from 'modules/issues/redux/api/callApi';
import { ReduxContext } from '../../redux/context';
import { issueServicesContext } from '../IssueServicesContext';
import { IssueIdContext } from '../Issue/IssueIdContext';

export const CallTo: FC<CallToProps> = (props) => {
  const { phoneData, children } = props;
  const { phoneE164, id } = phoneData;

  const redux = useContext(ReduxContext);
  const { callAvailableService, callMetaService } = useContext(issueServicesContext);
  const issueId = useContext(IssueIdContext);
  const dispatch = useDispatch();

  const saveCall = redux.slices.timelineSlice.asyncActions.saveCall;

  const callId = useRef<number>();
  const manager = useRef<WebphoneEventManager>();

  const destroy = useCallback(() => {
    if (manager.current) {
      manager.current.destroy();
      manager.current = undefined;
    }
  }, []);

  const handleCallInit = useCallback<OnCallInit>(
    ({ webphone, meta }) => {
      manager.current = new WebphoneEventManager(webphone, {
        [WebphoneOutgoingEventKind.NewOutgoingCall]: (event) => {
          dispatch(
            saveCall({
              callId: event.callId,
              direction: CallDirection.Out,
              kikId: id,
              status: CallStatus.InProgress,
              operatorPhoneNumber: meta.caller,
              issueId,
            }),
          ).then((data) => {
            webphoneStateManager.setCrmCallId(data.data?.id);
            callId.current = data.data?.id;
            return data;
          });
        },
        [WebphoneOutgoingEventKind.EstablishedCall]: () => {
          if (callId.current) {
            establishedCallEventListener(callId.current);
          }
        },
        [WebphoneOutgoingEventKind.CallEnd]: (event) => {
          if (callId.current) {
            dispatch(
              saveCall({
                id: callId.current,
                callId: event.callId,
                direction: CallDirection.Out,
                kikId: id,
                status: CallStatus.End,
                operatorPhoneNumber: meta.caller,
                issueId,
              }),
            );
          }

          destroy();
        },
      });
    },
    [dispatch, saveCall, id, issueId, destroy],
  );

  return (
    <BaseCallTo
      callMetaService={callMetaService}
      callAvailableService={callAvailableService}
      callee={phoneE164}
      onCallInit={handleCallInit}
      onCallError={destroy}
    >
      {children}
    </BaseCallTo>
  );
};
