import React, { useCallback, useRef } from 'react';
import Bluebird from 'bluebird';
import { Form, Field } from 'react-final-form';
import { connect } from 'react-redux';
import {
  withReduxContext,
  InjectedReduxContextProps,
  CreateIssueDataSome,
} from 'modules/issues/redux';
import Button from '@crm/components/dist/lego2/Button';
import TextInput from 'lego/final-form/TextInput';
import cx from 'classnames';
import createI18N from '@yandex-int/i18n';
import * as commonKeyset from 'common.i18n';
import * as keyset from './CreateNewIssue.i18n';
import css from './IssueNew.module.css';

const i18nCreate = createI18N(commonKeyset)('create');
const i18nNewIssue = createI18N(keyset)('newIssue');

interface FormData {
  name: string;
}

interface Props {
  getSubmitParamExtension?: () => { accountId?: number; parentId?: number };
  className?: string;
}

interface ConnectedDispatch {
  onSubmit: (values: FormData) => Bluebird<CreateIssueDataSome>;
}

const initialValues: FormData = {
  name: (undefined as unknown) as string,
};

const IssueNew: React.FC<Props & ConnectedDispatch & InjectedReduxContextProps> = (props) => {
  const { onSubmit, className } = props;

  const ref = useRef<HTMLInputElement | null>(null);
  const setRef = useCallback((node: HTMLInputElement) => {
    if (node != null) {
      ref.current = node;
    }
  }, []);

  return (
    <Form
      initialValues={initialValues}
      onSubmit={(values, formApi) => {
        return onSubmit(values).then((data) => {
          setTimeout(formApi.reset, 0);

          if (ref.current) {
            ref.current.focus();
          }

          return data;
        });
      }}
    >
      {(formApi) => (
        <form className={cx(css.b, className)} onSubmit={formApi.handleSubmit}>
          <Field
            innerRef={setRef}
            component={TextInput}
            pin="round-clear"
            name="name"
            placeholder={i18nNewIssue}
          />
          <Button
            pin="clear-round"
            type="submit"
            disabled={!formApi.dirty || formApi.submitting}
            view="action"
            progress={formApi.submitting}
          >
            {i18nCreate}
          </Button>
        </form>
      )}
    </Form>
  );
};

const Form2 = withReduxContext<Props>(
  connect<null, ConnectedDispatch, Props & InjectedReduxContextProps>(null, (dispatch, props) => ({
    onSubmit: (values) => {
      let resultValues = values;
      if (props.getSubmitParamExtension) {
        resultValues = { ...values, ...props.getSubmitParamExtension() };
      }
      return dispatch(props.redux.slices.issueSlice.asyncActions.createIssue(resultValues));
    },
  }))(IssueNew),
);

export default Form2;
