import React, { FC, useCallback, useContext } from 'react';
import { connect, useDispatch } from 'react-redux';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import { XivaContext } from 'modules/xiva';
import { config } from 'services/Config';
import FilterTree from './FilterTree';
import css from './Filters.module.css';
import { TreeRoot } from './utils/treeWalker';
import { TreeRootAdapter, TreeStructure, TreeData } from './utils/TreeRootAdapter';
import { useXivaRefresh } from './useXivaRefresh';
import { TreeViewFilter } from './TreeViewFilter';

interface ConnectedState {
  treeStructure: TreeStructure;
  treeData: TreeData;
}

type Props = ConnectedState & InjectedReduxContextProps;

const Filters: FC<Props> = ({ treeStructure, treeData, redux }) => {
  const dispatch = useDispatch();
  const xiva = useContext(XivaContext);

  const hasFilter = useCallback(
    (id: number) => {
      return Boolean(treeData[id]);
    },
    [treeData],
  );

  const getFilterById = useCallback(
    (id: number) => {
      return treeData[id];
    },
    [treeData],
  );

  const updateFilters = useCallback(
    (update: Record<number, { counter: number }>) => {
      dispatch(
        redux.slices.storageSlice.actions.update({
          issueFilters: {
            ...treeData,
            ...update,
          },
        }),
      );
    },
    [dispatch, treeData, redux.slices.storageSlice.actions],
  );

  useXivaRefresh({
    xiva,
    has: hasFilter,
    getById: getFilterById,
    update: updateFilters,
  });

  if (!treeStructure || !treeData) {
    return null;
  }

  const treeRoot: TreeRoot = new TreeRootAdapter(treeStructure, treeData);
  const isNewDesign = config.value.features.newDesignFilters;

  return (
    <div className={css.b}>
      {isNewDesign ? <TreeViewFilter treeRoot={treeRoot} /> : <FilterTree treeRoot={treeRoot} />}
    </div>
  );
};

export default withReduxContext(
  connect(
    (state, props: InjectedReduxContextProps) =>
      (({
        treeStructure: props.redux.selectors.storage.nodes.getAll(state, 'issueFilters'),
        treeData: props.redux.selectors.storage.issueFilters.getAll(state),
      } as unknown) as ConnectedState),
  )(Filters),
);
