import React, { useCallback } from 'react';
import { Form, FieldRow, errorMessage } from 'components/FinalForm';
import * as yup from 'yup';
import { issueUrlRegExp, getIssueIdFormIssueUrl } from 'utils/issueUrl';
import { LinkFormProps, issueLinkRoleItems } from './types';
import { IssueLinkRole } from '../../../types';

const errorText = 'Номер или ссылка на тикет';

const schema = yup.object().shape({
  targetId: yup
    .string()
    .matches(issueUrlRegExp, errorText)
    .required(errorMessage.required),
});

const LinkForm: React.FC<LinkFormProps> = (props) => {
  const { issueId, size, onSubmit, ...other } = props;

  const handleSubmit = useCallback(
    (values, formikBag) => {
      const formattedValues = {
        issueId: values.issueId,
        targetId: getIssueIdFormIssueUrl(values.targetId),
        linkRole: values.linkRole,
      };

      onSubmit(formattedValues, formikBag);
    },
    [onSubmit],
  );

  return (
    <Form
      initialValues={{ issueId, linkRole: IssueLinkRole.Related }}
      validationSchema={schema}
      title="Добавить связь"
      size={size}
      onSubmit={handleSubmit}
      {...other}
    >
      <FieldRow
        name="linkRole"
        component="Select"
        label="Тип связи"
        size={size}
        inputProps={{
          options: issueLinkRoleItems,
          hasEmptyValue: false,
        }}
      />
      <FieldRow name="targetId" component="TextInput" label="Тикет" size={size} />
    </Form>
  );
};

export default LinkForm;
