import React, { useEffect, useMemo, useRef, useState } from 'react';
import { Dispatch } from 'redux';
import { connect } from 'react-redux';
import { observer } from 'mobx-react-lite';
import { InjectedReduxContextProps, withReduxContext } from 'modules/issues/redux';
import ReactBuilder from 'components/ReactBuilder';
import Access from 'utils/Access';
import { ReactTabElement, Tab, Tabs, ThemeType } from 'lego/components/Tabs';
import cx from 'classnames';
import { config } from 'services/Config';
import { IssueList } from 'modules/issues';
import { ExpandWidget } from 'modules/categorization2';
import { TableWithFilters } from 'components/Table';
import { DynamicallyModalForm } from 'components/DynamicallyModalForm';
import { IssueHeader } from 'components/IssueHeader';
import { ExpandButton } from 'components/ExpandButton';
import { FactorStreamProvider } from 'services/FactorService/FactorService';
import { EObjectFormatter } from 'utils/EObjectFormatter';
import Spinner from 'components/Spinner';
import { ExpandWidgetNewIssue } from '../ExpandWidgetNewIssue';
import { MailFileInputContext, FileInputApi, MessageFileInputContext } from '../FileInputContext';
import IssuePanel from './IssuePanel';
import IssuePropsContext from './IssuePropsContext';
import css from './Issue.module.css';
import { schemeMain } from '../schemes';
import { newTimelineDescription } from '../schemes/newTimelineDescription';
import { components } from './config';
import { Sort, Timeline } from '../Timeline';
import { TimelineV2 } from '../TimelineV2';
import IssueHistoryIframe from '../IssueHistoryIframe';
import { IssueObject } from '../../types';
import IssueLinks from './IssueLinks';
import IssueSimilar from './IssueSimilar';
import { IssueServicesProvider } from '../IssueServicesContext';
import { IssueIdContext } from './IssueIdContext';
import { useXivaRefresh } from './useXivaRefresh';

interface ConnectedState {
  issue: IssueObject;
  isFetch: boolean;
  isLoad: boolean;
  showTimeline: boolean;
  subIssuesAccess: number;
  showSimilarIssues: boolean;
  subIssuesCounter?: number;
  similarIssuesCounter?: number;
  productsCounter?: number;
  moduleName: string;
  showProductsTab?: boolean;
}

interface ConnectedDispatch {
  fetchIssue: () => void;
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  dispatch: Dispatch<any>;
}

export interface OwnProps {
  id: number;
  timelineId?: string;
  showAttributes?: boolean;
  showTabsHeader?: boolean;
  showHeader?: boolean;
  showSimilarIssues?: boolean;
  timelineHeight?: number;
  maxAccess?: number;
  forceUseFeatureIssueTimelineV2?: boolean;
  isInChatsPopup?: boolean;
  showCommentForm?: boolean;
  showMailForm?: boolean;
  showMessageForm?: boolean;
  showInternalMessageForm?: boolean;
  showExpandWidgets?: boolean;
}

type Props = ConnectedState & ConnectedDispatch & OwnProps;

const counterFormatter = (counter?: number) => {
  if (!counter) {
    return undefined;
  }

  return `(${counter})`;
};

const Issue: React.FC<Props> = observer((props: Props & InjectedReduxContextProps) => {
  const {
    issue,
    id,
    timelineId,
    isFetch,
    isLoad,
    showTimeline,
    showAttributes = true,
    showHeader = true,
    showTabsHeader = true,
    showExpandWidgets = true,
    subIssuesAccess,
    showSimilarIssues,
    maxAccess,
    fetchIssue,
    subIssuesCounter,
    similarIssuesCounter,
    productsCounter,
    moduleName,
    showProductsTab,
    timelineHeight,
    forceUseFeatureIssueTimelineV2,
    showCommentForm,
    showMailForm,
    showMessageForm,
    showInternalMessageForm,
    isInChatsPopup = false,
  } = props;

  if (config.value.features.enableInternalChats) {
    //TODO: при выпиливании фичи enableInternalChats нужно в schemeMain заменить SupportChat на SupportChatIssue и удалить строку ниже
    schemeMain.common.content[1].content[1].content[1].content[1].content[1].props.component =
      'SupportChatIssue';
  }

  const issueTimelineV2 = useMemo(
    () => forceUseFeatureIssueTimelineV2 || config.value.features.issueTimelineV2,
    [forceUseFeatureIssueTimelineV2, config.value.features.issueTimelineV2],
  );

  const [expandDescription, setExpandDescription] = useState(false);
  const [showExpandButton, setShowExpandButton] = useState(true);

  useEffect(() => {
    if (id) {
      fetchIssue();
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [id]);

  useXivaRefresh(id);
  const mailInputRef = useRef<FileInputApi>(null);
  const messageInputRef = useRef<FileInputApi>(null);

  const handleDescriptionExpand = () => {
    setExpandDescription((expand) => !expand);
  };

  const handleTabChange = (prevValue, nextValue) => {
    setShowExpandButton(nextValue === 0);
  };

  if (isLoad && issue) {
    const tabs: ReactTabElement[] = [];

    if (!issueTimelineV2) {
      tabs.push(
        <Tab key="Текущий" title="Текущий" keepMount className={css.b__currentTab}>
          <span data-testid="issue-description" style={{display: 'contents'}}>
            <ReactBuilder components={components} meta={schemeMain.main} />
          </span>
          {showTimeline && (
            <div className={css.b__blockWrap}>
              <div className={css.b__block}>
                <div className={css.b__row}>
                  <span className={css.b__h2}>Коммуникации</span>
                  <span className={css.b__right}>
                    <Sort />
                  </span>
                </div>
                <Timeline issueId={id} timelineId={timelineId} maxAccess={maxAccess} />
              </div>
              {showExpandWidgets && (
                <div className={css.b__expandWidget}>
                  <ExpandWidgetNewIssue />
                  <ExpandWidget key={id} className={css.b__expandWidget} />
                </div>
              )}
            </div>
          )}
        </Tab>,
      );
    }

    if (issueTimelineV2) {
      tabs.push(
        <Tab testid="current-tab" key="Текущий" title="Текущий" keepMount className={css.b__newTimelineTab}>
          <div className={css.b__newTimelineWrap}>
            {expandDescription && (
              <div data-testid="issue-description" className={css.b__descriptionWrap}>
                <ReactBuilder components={components} meta={newTimelineDescription} />
              </div>
            )}
            <TimelineV2
              moduleName={moduleName}
              issueId={id}
              timelineId={timelineId}
              className={css.b__newTimeline}
              maxAccess={maxAccess}
              height={timelineHeight}
              forceUseFeatureIssueTimelineV2={issueTimelineV2}
              showCommentForm={showCommentForm}
              showMailForm={showMailForm}
              showMessageForm={showMessageForm}
              showInternalMessageForm={showInternalMessageForm}
              afterContent={showExpandWidgets && <ExpandWidgetNewIssue />}
            />
            {showExpandWidgets && (
              <ExpandWidget key={id} className={css.b__expandWidgetForNewTimeline} />
            )}
          </div>
        </Tab>,
      );
    }

    if (showProductsTab) {
      tabs.push(
        <Tab key="Продукты" title="Продукты" buttonAfterContent={counterFormatter(productsCounter)}>
          <TableWithFilters
            showExtendsFilters={false}
            url={`/v0/blocks/issueOpportunityProduct/${id}`}
          />
        </Tab>,
      );
    }

    if (Access.isRead(subIssuesAccess)) {
      tabs.push(
        <Tab
          key="Связанные"
          title="Связанные"
          buttonAfterContent={counterFormatter(subIssuesCounter)}
          testid="linked-issues-tab"
        >
          <IssueLinks issueId={id} maxAccess={maxAccess} access={subIssuesAccess} />
        </Tab>,
      );
    }

    if (__DEV_FEATURE__ || showSimilarIssues) {
      tabs.push(
        <Tab
          key="Похожие"
          title="Похожие"
          buttonAfterContent={counterFormatter(similarIssuesCounter)}
          testid="similar-issues-tab"
        >
          <IssueSimilar issueId={id} maxAccess={maxAccess} />
        </Tab>,
      );
    }

    tabs.push(
      <Tab testid="history-action-tab" key="История изменений" title="История изменений" className={css.b__historyTab}>
          <IssueHistoryIframe issueId={id} className={css.b__history} />
      </Tab>,
    );

    if (issue.data.account?.id != null && Access.isRead(issue.props.accountTasks?.access)) {
      const accountId = issue.data.account.id;
      const paramExtension = () => ({ accountId });
      tabs.push(
        <Tab testid="tasks-tab" key="Задачи" title="Задачи">
          <IssueList
            hash={accountId}
            hasInfiniteScroll
            getParamExtension={paramExtension}
            showNewIssueForm={Access.isEdit(issue.props.accountTasks?.access)}
            maxAccess={maxAccess} // TODO: https://st.yandex-team.ru/CRM-12892
          />
        </Tab>,
      );
    }

    const issueHeaderV2 = config.value.features.issueHeaderV2;
    const className = cx(css.b, {
      [css.b_newHeader]: issueHeaderV2,
      [css.b_chatsPopup]: isInChatsPopup,
    });

    return (
      <FactorStreamProvider>
        <MailFileInputContext.Provider value={mailInputRef}>
          <MessageFileInputContext.Provider value={messageInputRef}>
            <IssueServicesProvider issue={issue}>
              <IssuePropsContext.Provider value={props}>
                <IssueIdContext.Provider value={id}>
                  <div data-testid="issue" className={className}>
                    <div className={css.b__main}>
                      {!issueHeaderV2 && showHeader && (
                        <div data-testid="issue-header" className={css.b__top}>
                          <ReactBuilder components={components} meta={schemeMain.common} />
                        </div>
                      )}
                      {issueHeaderV2 && showHeader && <IssueHeader issue={issue} />}
                      <Tabs
                        key={id}
                        testid="issue-header-tabs"
                        theme={issueHeaderV2 ? ThemeType.MGBottom : ThemeType.Top}
                        navigationTheme="start"
                        className={css.b__tabs}
                        classNameHeader={css.b__stabsHeader}
                        tone="grey"
                        onDidUpdate={handleTabChange}
                        showHeader={showTabsHeader}
                        headerRight={
                          issueTimelineV2 &&
                          showExpandButton && (
                            <ExpandButton
                              expanded={expandDescription}
                              onClick={handleDescriptionExpand}
                              className={cx(css.b__expandButton, {
                                [css.b__expandButton_expanded]: expandDescription,
                              })}
                              view="clear"
                              text="Описание"
                            />
                          )
                        }
                      >
                        {tabs}
                      </Tabs>
                    </div>
                    {showAttributes && <IssuePanel issue={issue} />}
                  </div>
                  <DynamicallyModalForm />
                </IssueIdContext.Provider>
              </IssuePropsContext.Provider>
            </IssueServicesProvider>
          </MessageFileInputContext.Provider>
        </MailFileInputContext.Provider>
      </FactorStreamProvider>
    );
  }

  if (isFetch) {
    return (
      <div>
        <Spinner visible modal={false} />
      </div>
    );
  }

  return null;
});

const ConnectedIssue = connect<
  ConnectedState,
  ConnectedDispatch,
  OwnProps & InjectedReduxContextProps
>(
  (state, props) => ({
    issue: props.redux.selectors.storage.issues.getItem(state, props.id),
    moduleName: props.redux.name,
    isLoad: props.redux.selectors.isIssueLoad(state, props.id),
    isFetch: props.redux.selectors.isIssueFetch(state, props.id),
    showTimeline: Access.isRead(
      props.redux.selectors.storage.issues.getFieldAccess(state, props.id, 'timeline'),
    ),
    subIssuesAccess:
      props.redux.selectors.storage.issues.getFieldAccess(state, props.id, 'subIssues') || 0,
    showSimilarIssues:
      Boolean(props.showSimilarIssues) &&
      Access.isRead(
        props.redux.selectors.storage.issues.getFieldAccess(state, props.id, 'possibleDuplicates'),
      ),
    subIssuesCounter: props.redux.selectors.storage.issues.getFieldCounter(
      state,
      props.id,
      'subIssues',
    ),
    showProductsTab: Access.isRead(
      props.redux.selectors.storage.issues.getFieldAccess(state, props.id, 'issueProducts'),
    ),
    productsCounter: props.redux.selectors.storage.issues.getFieldCounter(
      state,
      props.id,
      'issueProducts',
    ),
    similarIssuesCounter: props.redux.selectors.storage.issues.getFieldCounter(
      state,
      props.id,
      'possibleDuplicates',
    ),
  }),
  (dispatch, props) => ({
    fetchIssue: () => {
      dispatch(props.redux.slices.issueSlice.asyncActions.fetchIssue(props.id));
    },
    dispatch,
  }),
)(Issue);

/* TODO: refactoring @ts-ignore */
// @ts-ignore
ConnectedIssue.defaultProps = {
  showSimilarIssues: true,
};

export default withReduxContext(ConnectedIssue);
