import React, { FC, memo, useCallback, useContext, useMemo, useRef } from 'react';
import {
  AttributesScheme,
  AttributesSchemeRow,
  ComponentsProperty,
} from '@crm/components/dist/AttributesByScheme';
import {
  AttributesByScheme,
  GrantsChangeHandler,
  SelectChangeHandler,
  SkillsChangeHandler,
  StaffUserInputChangeHandler,
  SuggestChangeHandler,
  TagsChangeHandler,
  TimersChangeHandler,
  GetCategorization,
  GetAccountInput,
} from 'components/AttributesByScheme';
import rum from 'services/Rum';
import { get } from 'entry/api/common';
import { useDispatch } from 'react-redux';
import Access from 'utils/Access';
import { config } from 'services/Config';
import { logger } from 'services/Logger';
import ArrayValueManager from 'utils/ArrayValueManager';
import { ETypeString } from 'types/entities';
import IssuePropsContext from './IssuePropsContext';
import { useCategorizationInitialization } from '../IssueAttribute/CategorizationAttribute/useCategorizationInitialization';

const MAP_ADD = {
  followers: 'actionFollowerAdd',
  tags: 'actionTagAdd',
  timers: 'editTimer',
};

const MAP_REMOVE = {
  followers: 'actionFollowerRemove',
  tags: 'actionTagRemove',
  timers: 'deleteTimer',
};

const areTimestampsTooFrequent = (timestamps: [number, number, number]): boolean => {
  if (timestamps.some((value) => value === -1)) {
    return false;
  }

  const statements = [timestamps[0] > timestamps[1] - 3000, timestamps[1] > timestamps[2] - 3000];
  return statements.every((value) => value);
};

export const IssueAttributesGrid: FC<{ scheme: AttributesScheme }> = memo((props) => {
  const { scheme } = props;

  const issueProps = useContext(IssuePropsContext);
  const redux = issueProps.redux;
  const issueId = issueProps.issue.id;
  const issueData = issueProps.issue.data;
  const issueAccesses = issueProps.issue.props;
  const fetchMutationIssue = redux.slices.issueSlice.asyncActions.fetchMutationIssue;
  const dispatch = useDispatch();

  const changeTimestampsRef = useRef<[number, number, number]>([-1, -1, -1]);
  const EObject = useMemo(() => ({ etype: ETypeString.Issue, eid: issueId }), [issueId]);

  const getCommon = useCallback(
    (attribute: AttributesSchemeRow) => ({
      value: issueData[attribute.name],
      isDisabled: !Access.isEdit(attribute.access || issueAccesses[attribute.name].access),
      EObject,
    }),
    [issueData, issueAccesses],
  );

  const change = useCallback(
    async (payload: { name: string; operation: string; value: unknown }) => {
      return dispatch(fetchMutationIssue(issueId, payload));
    },
    [issueId, fetchMutationIssue],
  );

  const commonChange = useCallback(async (attribute: AttributesSchemeRow, value: unknown) => {
    return change({
      name: attribute.backendUpdateKey || attribute.name,
      operation: 'change',
      value,
    });
  }, []);

  const handleGrantsChange = useCallback<GrantsChangeHandler>(
    async (attribute, value) => {
      const prevValue = getCommon(attribute).value;
      const action = ArrayValueManager.createAction(prevValue, value);
      const name =
        action.type === 'add'
          ? MAP_ADD[attribute.backendUpdateKey || attribute.name] ||
            attribute.backendUpdateKey ||
            attribute.name
          : MAP_REMOVE[attribute.backendUpdateKey || attribute.name] ||
            attribute.backendUpdateKey ||
            attribute.name;

      await change({
        name,
        operation: action.type,
        value: action.item.id,
      });
    },
    [change, getCommon],
  );

  const handleSelectChange = useCallback<SelectChangeHandler>(
    (attribute, value) => {
      return commonChange(attribute, value && value.id);
    },
    [commonChange],
  );

  const handleSkillsChange = useCallback<SkillsChangeHandler>(
    async (attribute, newValue) => {
      const prevValue = getCommon(attribute).value;
      const name = attribute.backendUpdateKey || attribute.name;

      if (newValue.length === prevValue.length) {
        const updatedSkill = newValue.find(
          (newSkill) => prevValue.find((v) => v.id === newSkill.id).value !== newSkill.value,
        );

        await change({
          name,
          operation: 'update',
          value: {
            id: updatedSkill?.id,
            name: updatedSkill?.text,
            value: updatedSkill?.value,
          },
        });
        return;
      }

      const action = ArrayValueManager.createAction(prevValue, newValue);
      await change({
        name,
        operation: action.type === 'delete' ? 'remove' : action.type,
        value: action.item,
      });
    },
    [change, getCommon],
  );

  const handleStaffUserInputChange = useCallback<StaffUserInputChangeHandler>(
    (attribute, value) => {
      return commonChange(attribute, value && value.id);
    },
    [commonChange],
  );

  const handleSuggestChange = useCallback<SuggestChangeHandler>(
    (attribute, value) => {
      return commonChange(attribute, value ? value.id : value);
    },
    [commonChange],
  );

  const handleTagsChange = useCallback<TagsChangeHandler>(
    (attribute, newValue) => {
      const prevValue = getCommon(attribute).value;
      const action = ArrayValueManager.createAction(prevValue, newValue);

      const name =
        action.type === 'add'
          ? MAP_ADD[attribute.backendUpdateKey || attribute.name] ||
            attribute.backendUpdateKey ||
            attribute.name
          : MAP_REMOVE[attribute.backendUpdateKey || attribute.name] ||
            attribute.backendUpdateKey ||
            attribute.name;

      return change({
        name,
        operation: action.type,
        value: action.item.id,
      });
    },
    [change, getCommon],
  );

  const handleTimersChange = useCallback<TimersChangeHandler>(
    async (attribute, newValue) => {
      const prevValue = getCommon(attribute).value[0] || null;
      const map = newValue ? MAP_ADD : MAP_REMOVE;
      const name =
        map[attribute.backendUpdateKey || attribute.name] ||
        attribute.backendUpdateKey ||
        attribute.name;
      if (newValue) {
        await change({
          name,
          operation: 'add',
          value: {
            ...newValue,
            id: prevValue?.id,
          },
        });
      } else {
        await change({
          name,
          operation: 'remove',
          value: prevValue?.id,
        });
      }
    },
    [change, getCommon],
  );

  const initialization = useCategorizationInitialization();
  const getCategorization = useCallback<GetCategorization>(
    (attribute) => ({
      onInitialize: initialization,
      url: attribute?.fieldProps?.provider,
    }),
    [initialization],
  );

  const issue = useContext(IssuePropsContext);
  const pinButtonAccess = issue.issue.props?.accessEnterPin?.access ?? 3;
  const accountAccesses = (issue.issue.props.account as unknown) as {
    accessNullAccount: number;
    accessCreateAccount: number;
    accessCreateLead: number;
  };
  const getAccountInput = useCallback<GetAccountInput>(
    (attribute) => ({
      ...getCommon(attribute),
      featureFlags: {
        canShowPinButton: Access.isEdit(pinButtonAccess),
        canShowNullClientButton: Access.isEdit(accountAccesses.accessNullAccount),
        canShowCreateButton: Access.isEdit(accountAccesses.accessCreateAccount),
        canShowLeadButton: Access.isEdit(accountAccesses.accessCreateLead),
      },
    }),
    [getCommon, accountAccesses],
  );

  const components: ComponentsProperty = {
    Categorization: {
      getProps: getCategorization,
    },
    AccountInput: {
      getProps: getAccountInput,
    },
    Grants: {
      changeHandler: handleGrantsChange,
    },
    Select: {
      getProps: (attribute) => ({
        loadItems: async () => {
          const url = attribute?.fieldProps?.provider;
          const items = attribute?.fieldProps?.items;
          if (Array.isArray(items)) {
            return items;
          }

          if (url) {
            const response = await get({
              url,
            });
            return response.items;
          }

          return [];
        },
      }),
      changeHandler: handleSelectChange,
    },
    Skills: {
      changeHandler: handleSkillsChange,
    },
    StaffUserInput: {
      changeHandler: handleStaffUserInputChange,
    },
    Suggest: {
      getProps: (attribute) => ({
        canDelete: attribute?.fieldProps?.canDelete,
        url: attribute?.fieldProps?.provider,
      }),
      changeHandler: handleSuggestChange,
    },
    Tags: {
      getProps: (attribute) => ({
        url: attribute?.fieldProps?.provider,
      }),
      changeHandler: handleTagsChange,
    },
    DateTimePicker: {
      getProps: (attribute) => ({
        minDate: attribute?.fieldProps?.minDate,
        canShowTime: attribute?.fieldProps?.time,
      }),
    },
    Timers: {
      changeHandler: handleTimersChange,
    },
  };

  return (
    <AttributesByScheme
      scheme={scheme}
      getCommon={getCommon}
      onCommonChange={commonChange}
      components={components}
      onEvent={(attribute, event) => {
        const timestamp = Date.now();

        rum.sendTimeMark('UX_attributeEventsNew', timestamp, {
          attributeName: attribute?.name,
          attributeComponent: attribute?.component,
          time: timestamp,
          ...event,
        });

        const login = config.value.user.login;
        if (
          event.type !== 'change' ||
          !login ||
          (login !== 'elena0708' && login !== 'nazarova-rush')
        ) {
          return;
        }

        const changeTimestamps = changeTimestampsRef.current;
        changeTimestamps[0] = changeTimestamps[1];
        changeTimestamps[1] = changeTimestamps[2];
        changeTimestamps[2] = timestamp;

        if (areTimestampsTooFrequent(changeTimestamps)) {
          logger.reportError(new Error('Too frequent attribute change events'), undefined);
        }
      }}
    />
  );
});
