import React, { ReactNode, PureComponent, ComponentType } from 'react';
import { config } from 'services/Config';
import { ContactData, Account } from 'types';
import { Contacts, CallToComponentType } from 'components/Contact';
import { Tabs, Tab, ThemeType } from 'lego/components/Tabs';
import ContactCreateButton from 'modules/contactCreateButton';
import { ContactComponentProps } from 'components/Contact/Contacts.types';
import Contact from './Contact';
import css from './IssueContactsPanel.module.css';
import { CallTo } from '../CallTo/CallTo';
import { CallToUseLogBroker } from '../CallToUseLogBroker';

interface Props {
  handContacts?: ContactData[];
  importContacts?: ContactData[];
  account?: Account;
  canCall: boolean;
  canEdit: boolean;
}

interface State {
  handContacts: ContactData[];
  importContacts: ContactData[];
}

interface RenderTabProps {
  contacts?: ContactData[];
  title: React.ReactNode;
  canCall: boolean;
  canEdit: boolean;
  contactComponent: ComponentType<ContactComponentProps>;
}

const callToEmptyComponent = () => null;

const renderTab = (props: RenderTabProps) => {
  const { title, contacts, contactComponent, canCall, canEdit } = props;

  let count: string | number = Number(contacts && contacts.length);

  if (count > 99) {
    count = '99+';
  }

  count = `(${count})`;

  let callToComponent: CallToComponentType | undefined = undefined;
  if (!canCall) {
    callToComponent = callToEmptyComponent;
  } else if (config.value.features.useYaCallsLB) {
    callToComponent = CallToUseLogBroker;
  } else if (config.value.features.useYaCalls) {
    callToComponent = CallTo;
  }

  return (
    <Tab title={title} buttonAfterContent={count} className={css.b__tab}>
      <Contacts
        contacts={contacts}
        callToComponent={callToComponent}
        contactComponent={contactComponent}
        canEdit={canEdit}
      />
    </Tab>
  );
};

const FORM_NAME = 'issueContactCreate';

class IssueContactsPanel extends PureComponent<Props, State> {
  public constructor(props: Props) {
    super(props);
    this.state = {
      handContacts: props.handContacts || [],
      importContacts: props.importContacts || [],
    };
  }

  private handleDelete = (id: number) => {
    let handContacts = this.state.handContacts.slice();
    for (let i = 0, length = handContacts.length; i < length; i++) {
      if (handContacts[i].contactId !== id) {
        continue;
      }

      handContacts.splice(i, 1);
      break;
    }
    this.setState({
      handContacts,
    });
  };

  private handleEdit = (contact) => {
    let handContacts = this.state.handContacts.slice();
    for (let i = 0, length = handContacts.length; i < length; i++) {
      if (handContacts[i].contactId !== contact.contactId) {
        continue;
      }

      handContacts.splice(i, 1, contact);
      break;
    }
    this.setState({
      handContacts,
    });
  };

  private handleCreateSuccess = (contact) => {
    this.setState((state) => ({
      handContacts: [...state.handContacts, contact],
    }));
  };

  private Contact = (props: ContactComponentProps) => {
    return (
      <Contact
        {...props}
        formName={FORM_NAME}
        accountId={(this.props.account && this.props.account.id) as number}
        onEdit={this.handleEdit}
        onDelete={this.handleDelete}
      />
    );
  };

  public render(): ReactNode {
    const { props } = this;
    const { account, canCall, canEdit } = props;
    let accountId: number | undefined;
    if (account) {
      accountId = account.id;
    }

    return (
      <div data-testid="issue-account-contacts" className={css.b}>
        {canEdit && (
          <div className={css.b__header}>
            <ContactCreateButton
              formName={FORM_NAME}
              accountId={accountId}
              onSubmitSuccess={this.handleCreateSuccess}
            />
          </div>
        )}
        <Tabs
          theme={ThemeType.Top}
          navigationTheme="start"
          tone="grey"
          className={css.b__tabs}
          classNameHeader={css.b__tabsHeader}
        >
          {renderTab({
            contactComponent: this.Contact,
            contacts: this.state.handContacts,
            title: 'Ручные',
            canCall,
            canEdit,
          })}
          {renderTab({
            contactComponent: this.Contact,
            contacts: this.state.importContacts,
            title: 'Из сервисов',
            canCall,
            canEdit: false,
          })}
        </Tabs>
      </div>
    );
  }
}

export default IssueContactsPanel;
