import React, { useCallback, CSSProperties } from 'react';
import { Dispatch } from 'redux';
import { connect } from 'react-redux';
import ReactBuilder from 'components/ReactBuilder';
import { Node } from 'components/ReactBuilder/types';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import { logger } from 'services/Logger';
import safeStringify from 'fast-safe-stringify';

import cx from 'classnames';

import { schemeMainListItem } from '../schemes';
import IssuePropsContext from './IssuePropsContext';
import css from './IssueItem.module.css';
import { components } from './config';
import EventFromField from '../utils/EventFromField';
import { IssueObject, IssueType, ModuleName, StorageData } from '../../types';
import { Node as StorageNode } from '../../types';

export interface OwnProps {
  id: number;
  selected?: boolean;
  scheme?: Node;
  className?: string;
  theme?: string;
  target?: string;
  style?: CSSProperties;
  node?: StorageNode; // FIXME CRM-16938: TEMP FOR LOG
}

interface ConnectedDispatch {
  dispatch: Dispatch<unknown>;
}

interface ConnectedState {
  issue: IssueObject;
  issues: StorageData['issues']; // FIXME CRM-16938: TEMP FOR LOG
}

const IssueItem: React.FC<OwnProps &
  ConnectedDispatch &
  ConnectedState &
  InjectedReduxContextProps> = (props) => {
  const { issue, selected, target, redux, className, scheme, theme, style } = props;

  const handleClick = useCallback(
    (e) => {
      if (!issue?.id) {
        return;
      }

      if (!EventFromField.isPrevent(e.target)) {
        redux.slices.issueSlice.goToIssue({
          moduleName: ModuleName[IssueType[issue.data.typeId]],
          issueId: issue.id,
          blank: target === '_blank',
        });
      }
    },
    [issue?.data.typeId, issue?.id, redux.slices.issueSlice, target],
  );

  // FIXME CRM-16938: TEMP FOR LOG
  if (!issue) {
    const log = {
      id: props.id,
      issues: Object.keys(props.issues ?? {}),
      node: props.node,
    };

    logger.reportError(new Error('Issue is undefined'), undefined, safeStringify(log));

    return null;
  }

  const classNames = cx(css.b, { [css.b_active]: selected }, className);

  return (
    <IssuePropsContext.Provider value={props}>
      <div
        role="presentation"
        tabIndex={-1}
        className={classNames}
        style={style}
        onClick={handleClick}
        data-testid={`IssueItem:${issue.id}`}
      >
        <div className={cx(css.b__wrap, css[`b__wrap_theme_${theme}`])}>
          <ReactBuilder components={components} meta={scheme} />
        </div>
      </div>
    </IssuePropsContext.Provider>
  );
};

IssueItem.defaultProps = {
  scheme: schemeMainListItem,
  theme: 'main',
};

export default withReduxContext<OwnProps>(
  connect<ConnectedState, ConnectedDispatch, OwnProps & InjectedReduxContextProps>(
    (state, props) => ({
      issue: props.redux.selectors.storage.issues.getItem(state, props.id),
      issues: props.redux.selectors.storage.issues.getAll(state), // FIXME CRM-16938: TEMP FOR LOG
    }),
  )(IssueItem),
);
