import React from 'react';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import { connect } from 'react-redux';
import Access from 'utils/Access';
import storagePathCreator from '../../utils/storagePathCreator';
import IssueLinksGroup from './IssueLinksGroup';
import IssueStartrekItem from './IssueStartrekItem';
import { IssueLinkRole } from '../../types';

interface ConnectedState {
  linksCount: number;
}

interface ConnectedDispatch {
  get: () => void;
}

interface ConnectedProps {
  issueId: number;
  access?: number;
  maxAccess?: number;
}

type Props = ConnectedDispatch & ConnectedProps & ConnectedState;

const config = [
  {
    id: 'subIssues',
    createPath: storagePathCreator.subIssues,
    title: 'Подзадачи',
    linkRole: IssueLinkRole.Child,
    useAccess: true,
  },
  {
    id: 'related',
    createPath: storagePathCreator.relatedIssues,
    title: 'Связанные',
  },
  {
    id: 'blocked',
    createPath: storagePathCreator.blockedIssues,
    title: 'Блокирует',
  },
  {
    id: 'blockedBy',
    createPath: storagePathCreator.blockedByIssues,
    title: 'Заблокирован',
  },
  {
    id: 'duplicate',
    createPath: storagePathCreator.duplicateIssues,
    title: 'Дубликаты',
  },
  {
    id: 'origin',
    createPath: storagePathCreator.originalIssues,
    title: 'Оригиналы',
  },
  {
    id: 'startrek',
    createPath: storagePathCreator.startrekIssues,
    title: 'Стартрек',
    component: IssueStartrekItem,
  },
];

class IssueLinks extends React.Component<Props> {
  public componentDidMount(): void {
    this.props.get();
  }

  public render() {
    const { issueId, access, maxAccess, linksCount } = this.props;

    if (!Access.isEdit(access) && !linksCount) {
      return <div>Связей нет</div>;
    }

    return (
      <div>
        {config.map(({ id, title, createPath, component, linkRole, useAccess }) => (
          <IssueLinksGroup
            key={id}
            title={title}
            issueId={issueId}
            path={createPath(issueId)}
            access={useAccess ? access : undefined}
            maxAccess={maxAccess}
            component={component}
            linkRole={linkRole}
          />
        ))}
      </div>
    );
  }
}

export default withReduxContext<ConnectedProps>(
  connect<ConnectedState, ConnectedDispatch, ConnectedProps & InjectedReduxContextProps>(
    (state, props) => ({
      linksCount: props.redux.selectors.getLinksCount(state, props.issueId),
    }),
    (dispatch, props) => ({
      get: () => {
        dispatch(
          props.redux.slices.nodesSlice.asyncActions.get({
            issueId: props.issueId,
          }),
        );
        dispatch(
          props.redux.slices.nodesSlice.asyncActions.getStartrekIssues({
            issueId: props.issueId,
          }),
        );
      },
    }),
  )(IssueLinks),
);
