import React, { useCallback } from 'react';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import { connect } from 'react-redux';
import Access from 'utils/Access';
import { IssueListView } from '../IssueList';
import css from './Issue.module.css';
import IssueItemLink from './IssueItemLink';
import { IssueLinkRole } from '../../types';

interface ConnectedProps {
  title: React.ReactNode;
  path: string;
  access?: number;
  maxAccess?: number;
  linkRole?: IssueLinkRole;
  issueId: number;
  component?: React.ComponentType;
}

interface ConnectedState {
  count: number;
}

type Props = ConnectedProps & ConnectedState;

const IssueLinksGroup: React.FC<Props> = (props) => {
  const {
    title,
    path,
    maxAccess,
    access,
    linkRole,
    issueId,
    count,
    component = IssueItemLink,
  } = props;

  const getLoadParamExtension = useCallback(() => ({ issueId }), [issueId]);
  const getSubmitParamExtension = useCallback(() => ({ linkedIssueId: issueId, linkRole }), [
    issueId,
    linkRole,
  ]);

  const canCreate = Access.isEdit(access);

  if (!canCreate && !count) {
    return null;
  }

  return (
    <div className={css.b__block}>
      <div className={css.b__row}>
        <span className={css.b__h2}>{title}</span>
      </div>
      <IssueListView
        hash={issueId}
        getLoadParamExtension={getLoadParamExtension}
        getSubmitParamExtension={getSubmitParamExtension}
        getParamExtension={getLoadParamExtension}
        path={path}
        showNewIssueForm={canCreate}
        maxAccess={maxAccess}
        component={component}
      />
    </div>
  );
};

export default withReduxContext<ConnectedProps>(
  connect<ConnectedState, null, ConnectedProps & InjectedReduxContextProps>((state, props) => {
    const node = props.redux.selectors.storage.nodes.getItem(state, props.path, 0) as
      | { items: unknown[] }
      | undefined;

    return {
      count: node?.items?.length || 0,
    };
  })(IssueLinksGroup),
);
