import React from 'react';
import Panel from 'modules/panel';
import { PanelScheme } from '@crm/components/dist/Panel';
import { IssueObject, IssueType } from 'modules/issues/types';
import Access from 'utils/Access';
import { config } from 'services/Config';
import { AttributesScheme } from 'components/AttributesByScheme';
import { PanelFiles, PanelEntity } from 'modules/clientHistory';
import { IssueAttribute } from '../IssueAttribute';
import css from './Issue.module.css';
import IssueContactsPanel from './IssueContactsPanel';
import { IssuePanelFactors } from './IssuePanelFactors';
import { IssueAttributesGrid } from './IssueAttributesGrid';

interface Props {
  issue: IssueObject;
}

const IssuePanel: React.FC<Props> = (props) => {
  const { issue } = props;
  const { account, id: issueId, typeId } = issue.data;
  const attributesScheme = issue.scheme && issue.scheme.attributes;
  const panels: PanelScheme[] = [];
  if (Array.isArray(attributesScheme)) {
    panels.push({
      id: 'attributes',
      title: 'Атрибуты',
      content: config.value.features.newDesignAttributes ? (
        <span data-testid="issue-attributes" style={{ display: 'contants' }}>
          <IssueAttributesGrid key={issueId} scheme={attributesScheme as AttributesScheme} />
        </span>
      ) : (
        <div data-testid="issue-attributes" className={css.b__attrs}>
          {attributesScheme!.map((field) => (
            <IssueAttribute className={css.b__attr} key={field.name} {...field} size="m" />
          ))}
        </div>
      ),
    });
  }

  if (account) {
    const accountId = account.id;
    const accountLogin = account.info.login;

    if (issue.props.upsale && Access.isRead(issue.props.upsale.access)) {
      const canEditUpsale = Access.isEdit(issue.props.upsale.access);

      panels.push({
        id: 'factors',
        title: 'Факторы',
        content: (
          <IssuePanelFactors
            accountId={accountId}
            accountLogin={accountLogin}
            canEditUpsale={canEditUpsale}
            issueId={issueId}
          />
        ),
      });
    }

    if (issue.props.contacts && Access.isRead(issue.props.contacts.access)) {
      panels.push({
        id: 'contacts',
        title: 'Контакты аккаунта',
        content: (
          <IssueContactsPanel
            key={(account || {}).id}
            account={account}
            canCall={
              Access.isEdit(issue.props.calls && issue.props.calls.access) &&
              typeId !== IssueType.Task
            }
            canEdit={Access.isEdit(issue.props.contacts.access)}
            handContacts={account.contacts && account.contacts.items}
            importContacts={account.importedKiks && account.importedKiks.items}
          />
        ),
        padding: false,
      });
    }

    panels.push({
      id: 'activity',
      title: 'История аккаунта',
      content: (
        <div className={css.b__accountHistory}>
          <PanelEntity
            key={accountId}
            param={{ clientId: accountId }}
            name="activity"
            loadOnMount
          />
        </div>
      ),
      keepMount: true,
      padding: false,
    });

    panels.push({
      id: 'files',
      title: 'Файлы аккаунта',
      content: (
        <div className={css.b__accountHistory}>
          <PanelFiles
            key={accountId}
            param={{ clientId: accountId }}
            name="files"
            loadOnMount
            showTitle={false}
          />
        </div>
      ),
      keepMount: true,
      padding: false,
    });
  }

  return (
    <Panel
      contentClassName={css.b__panel}
      maxWidth={600}
      minWidth={300}
      name="issue/attr"
      defaultPin
      defaultPanelId="attributes"
      panels={panels}
    />
  );
};

export default IssuePanel;
