import React from 'react';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import { connect } from 'react-redux';
import storagePathCreator from '../../utils/storagePathCreator';
import IssueLinksGroup from './IssueLinksGroup';

interface ConnectedState {
  count: number;
}

interface ConnectedDispatch {
  get: () => void;
}

interface ConnectedProps {
  issueId: number;
  maxAccess?: number;
}

type Props = ConnectedDispatch & ConnectedProps & ConnectedState;

const config = [
  {
    id: 'possibleDuplicateIssues',
    createPath: storagePathCreator.possibleDuplicateIssues,
    title: 'Возможные дубли',
  },
  {
    id: 'similarIssues',
    createPath: storagePathCreator.similarIssues,
    title: 'Похожие',
  },
];

class IssueSimilar extends React.Component<Props> {
  public componentDidMount(): void {
    this.props.get();
  }

  public render() {
    const { issueId, maxAccess, count } = this.props;

    if (!count) {
      return <div>Нет возможных дублей</div>;
    }

    return (
      <div>
        {config.map(({ id, title, createPath }) => (
          <IssueLinksGroup
            key={id}
            title={title}
            issueId={issueId}
            path={createPath(issueId)}
            maxAccess={maxAccess}
          />
        ))}
      </div>
    );
  }
}

export default withReduxContext<ConnectedProps>(
  connect<ConnectedState, ConnectedDispatch, ConnectedProps & InjectedReduxContextProps>(
    (state, props) => ({
      count: props.redux.selectors.getSimilarCount(state, props.issueId),
    }),
    (dispatch, props) => ({
      get: () => {
        dispatch(
          props.redux.slices.nodesSlice.asyncActions.get({
            issueId: props.issueId,
            loadPossibleDuplicates: true,
          }),
        );
      },
    }),
  )(IssueSimilar),
);
