import React from 'react';
import { connect } from 'react-redux';
import { withProps, compose } from 'recompose';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import IssueItem, { OwnProps } from './IssueItem';
import createTreeNode, { TreeNodeProps, ComponentProps } from '../createTreeNode';
import storagePathCreator from '../../utils/storagePathCreator';
import css from './IssueItem.module.css';
import { Node } from '../../types';

interface WrapProps {
  className?: string;
}

const IssueItemWrap: React.FC<WrapProps> = ({ className, children }) => (
  <div className={className}>{children}</div>
);

IssueItemWrap.defaultProps = {
  className: css.wrap,
};

interface ConnectedState {
  node: Node;
}

interface ConnectedProps {
  id: number;
  path: string;
}

const hoc = compose(
  withReduxContext,
  connect<ConnectedState, {}, ConnectedProps & InjectedReduxContextProps & TreeNodeProps>(
    (state, { id, path, redux }) => ({
      node: redux.selectors.storage.nodes.getItem(state, path, id),
      ...redux.selectors.meta.nodes(state, storagePathCreator.merge(path, id)),
    }),
    (dispatch, props) => ({
      toggle: e => {
        const { id, path } = props;
        e.stopPropagation();
        return dispatch(props.redux.slices.storageSlice.actions.toggle(path, id));
      },
      more: offset => e => {
        const { id, path, getLoadParamExtension } = props;

        if (e) {
          e.stopPropagation();
        }

        const loadParam = getLoadParamExtension ? getLoadParamExtension() : null;

        dispatch(
          props.redux.slices.nodesSlice.asyncActions.get({
            ...loadParam,
            parentId: id,
            offset,
            path,
            nodeId: storagePathCreator.merge(path, id),
          }),
        );
      },
    }),
  ),
  withProps(props => {
    const offset = ((props.node || {}).items || []).length;

    return {
      ...props,
      more: props.more(offset),
      isLoadingFirstPage: !props.offset,
    };
  }),
);

const IssueItemAdapter: React.FC<OwnProps & ComponentProps> = ({
  id,
  style,
  className,
  scheme,
  theme,
  target,
}) => (
  <IssueItem
    id={id}
    style={style}
    className={className}
    scheme={scheme}
    theme={theme}
    target={target}
  />
);

export default createTreeNode({
  component: IssueItemAdapter,
  wrap: IssueItemWrap,
  hoc,
});
