import React, { useCallback, MouseEvent } from 'react';
import { useDispatch } from 'react-redux';
import { setEObject } from 'modules/issuePreview/redux/previewSlice';
import { Type } from 'modules/issuePreview/types';
import CopyButton from 'lego/components/CopyButton';
import cx from 'classnames';
import createLink from 'modules/issues/utils/createLink';
import getRoutingRoot from 'utils/getRoutingRoot';
import { ModuleName, IssueType as IssueTypeEnum } from 'modules/issues/types';
import { getHashSearch } from 'utils/getHashSearch';
import { cnTheme } from '@yandex-lego/components/Theme';
import { crmTheme } from '@crm/components/dist/lego2/Theme/presets/themes';
import { dispatchUXEvent } from 'components/UXLogger';
import { withIssueProps, InjectedContextProps } from './IssuePropsContext';
import IssueFieldLayout, { Props as IssueFieldLayoutProps } from '../IssueFieldLayout';
import css from './IssueType.module.css';

interface Props {
  type: string;
  id: number;
  hasPreview?: boolean;
  issueUrl: string;
  moduleName: ModuleName;
}

const IssueType: React.FC<Props & IssueFieldLayoutProps> = ({
  type,
  id,
  size,
  weight,
  color,
  hasPreview,
  issueUrl,
  moduleName,
}) => {
  const dispatch = useDispatch();

  const onClick = useCallback(
    (e: MouseEvent<HTMLDivElement>) => {
      e.stopPropagation();
      // stopPropagation for native event on document
      e.nativeEvent.stopImmediatePropagation();

      dispatchUXEvent(e.type, undefined, e.target);

      dispatch(setEObject({ id, type: Type.Issue }));
    },
    [id, dispatch],
  );

  return (
    <IssueFieldLayout
      onClick={hasPreview ? onClick : undefined}
      size={size}
      weight={weight}
      color={color}
      className={cx(css.root, { [css.root_hasPreview]: hasPreview })}
    >
      {type} {id}{' '}
      {moduleName !== ModuleName.Issue && (
        <CopyButton
          copy={issueUrl}
          className={cx(css.root__copyButton, cnTheme({ root: crmTheme.root }))}
          iconClassName={css.root__copyButtonIcon}
          title="Скопировать ссылку"
          withTooltip
          tooltipCloseDelay={1500}
          size="xs"
          view="clear"
          stopPropagation
        />
      )}
    </IssueFieldLayout>
  );
};

const DataAdapter: React.FC<InjectedContextProps & IssueFieldLayoutProps> = (props) => {
  const { id, type, typeId } = props.issue.issue.data;
  const issueUrl =
    getRoutingRoot({ ...window.location, hash: '' }) +
    createLink({
      typeId,
      id,
    }) +
    getHashSearch(window.location.hash);

  const moduleName = ModuleName[IssueTypeEnum[typeId]];

  return <IssueType id={id} type={type} issueUrl={issueUrl} moduleName={moduleName} {...props} />;
};

export default withIssueProps<IssueFieldLayoutProps>(DataAdapter);
