import React, { useContext, useRef, useCallback } from 'react';
import get from 'lodash/get';
import { post } from 'entry/api/common';
import { Timeline } from 'modules/issues/components/Timeline';
import { EType } from 'types/entities';
import { IssueType } from 'modules/issues/types';
import { types } from 'modules/categorization2';
import { config } from 'services/Config';
import { TimelineV2 } from 'modules/issues/components/TimelineV2';
import { ModuleName } from 'modules/issues/types';
import { IssueIdContext } from '../../Issue/IssueIdContext';
import { IssuePropsContext } from '../../Issue/IssuePropsContext';
import { loadCategorization, loadTip, loadSearchResults, loadAlerts } from './utils';
import { createValueTreeFromIds } from '../../Timeline/Categorization/utils';
import css from './Widget.module.css';

export const useCategorizationInitialization = () => {
  const issueId = useContext(IssueIdContext);
  const issueProps = useContext(IssuePropsContext);
  const type: number = get(issueProps, 'issue.data.typeId');
  const treeDataRef = useRef<{
    [key: string]:
      | types.Category
      | {
          items: number[];
        };
  }>();

  const previewComponent = useCallback(() => {
    if (config.value.features.issueTimelineV2) {
      return (
        <TimelineV2
          view="categorization"
          className={css.Widget__timelineV2}
          issueId={issueId}
          moduleName={ModuleName.Ticket}
          maxAccess={1}
        />
      );
    }
    return <Timeline issueId={issueId} expandItems maxAccess={1} />;
  }, [issueId]);

  const handleSearch = (text: string) => {
    return loadSearchResults({
      issueId,
      workplace: 'Issue',
      text,
    });
  };

  const handleTipLoad = (id: number) => {
    return loadTip(id, issueId);
  };

  const handleTipComment: types.CommentFormExternalHandler = (payload) => {
    const { id, text } = payload;
    return post({
      url: '/actions/comment',
      global: false,
      data: {
        eType: EType.Category,
        eId: id,
        text,
      },
    });
  };

  const targetMeta = { id: issueId, type: IssueType[type], description: 'Тикет' };

  const initialization = useCallback(
    (props: {
      value?: {
        id: number;
        name: string;
      } | null;
      endpoint: string;
      onChange(value: { id: number } | null): void;
    }): types.ExternalData => {
      return {
        targetMeta,
        previewComponent,
        load: () => {
          return loadCategorization(props.endpoint, issueId).then(({ byId, root }) => {
            treeDataRef.current = {
              ...byId,
              root: {
                items: root,
              },
            };

            return {
              byId,
              root,
              highlighted: [],
              valueAsTree: createValueTreeFromIds(root, byId, props.value ? [props.value.id] : []),
            };
          });
        },
        search: handleSearch,
        loadTip: handleTipLoad,
        loadAlerts,
        commentSubmit: handleTipComment,
        save: (value?: { id: number }[]) => {
          if (!value || !value.length) {
            props.onChange(null);
            return;
          }

          if (!treeDataRef.current) {
            return;
          }

          props.onChange(value[0]);
        },
        changeStrategy: 'one-branch' as const,
      };
    },
    [],
  );

  return initialization;
};
