import { get, baseCallApi } from 'api/common';
import { normalize } from 'normalizr';
import { config } from 'services/Config';
import { ById } from 'modules/categorization2/types';
import { AlertDTO } from 'types/dto/AlertDTO';
import { rootSchema } from '../../Timeline/Categorization/schemas';
import { getDeepResults } from '../../Timeline/Categorization/utils';

export const loadCategorization = (
  url: string,
  issueId: number,
): Promise<{
  byId: ById;
  root: number[];
}> => {
  return get({
    url,
    data: {
      eId: issueId,
      eType: 'Issue',
    },
  }).then((response) => {
    const normalizedData = normalize(response, rootSchema);
    const byId = normalizedData.entities.items || {};
    const root = normalizedData.result.items || [];

    return {
      byId,
      root,
    };
  });
};

export const loadTip = (categoryId: number, issueId: number) => {
  return baseCallApi({
    url: `/view/category/${categoryId}/daasTip/Issue/${issueId}`,
    showBackendErrorForUser: false,
    global: false,
    headers: {
      Accept: 'text/html',
    },
  });
};

export const loadSearchResults = ({
  issueId,
  workplace,
  text,
}: {
  issueId: number;
  workplace: string;
  text: string;
}) => {
  return get({
    url: '/view/category/tree',
    global: false,
    data: {
      issueId,
      workplace,
      text,
    },
  }).then((data = {}) => {
    const normalizedData = normalize(data, rootSchema);
    const byId = normalizedData.entities.items || {};
    const ids = normalizedData.result.items || [];
    const highlightRangesById = Object.keys(byId).reduce(
      (acc, id) => ({
        ...acc,
        [id]: byId[id].highlightRanges,
      }),
      {},
    );

    return {
      resultIds: getDeepResults(ids, byId),
      highlightRangesById,
    };
  });
};

export const loadAlerts = (categoryIds: number[]): Promise<AlertDTO[]> => {
  return get({
    url: '/alerts/active',
    global: false,
    data: {
      categoryIds,
    },
  });
};
