import { connect } from 'react-redux';
import { WithOId } from 'types';
import { Dispatch } from 'redux';
import { Event } from 'types/Event';
import rum from 'services/Rum';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import {
  changeEvent,
  changeFailEvent,
  changeSuccessEvent,
  startEditingEvent,
  stopEditingEvent,
} from 'utils/events';
import { IssueAttributeWithContext } from './IssueAttributeWithContext';
import { issueAttributesAdapter } from './issueAttributesAdapter';
import { withIssueProps } from '../Issue/IssuePropsContext';
import withFieldProps, { WrappedComponentProps } from '../Issue/withFieldProps';

const MAP_ADD = {
  tags: 'actionTagAdd',
  followers: 'actionFollowerAdd',
  timers: 'editTimer',
};

const MAP_REMOVE = {
  tags: 'actionTagRemove',
  followers: 'actionFollowerRemove',
  timers: 'deleteTimer',
};

// hack for Select with isSingleValue
const formatValue = (value) => (value != null && typeof value === 'object' ? value.id : value);

interface Props extends InjectedReduxContextProps, WrappedComponentProps, WithOId {
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  value: any;
  backendUpdateKey?: string;
  name: string;
  access?: number;
  component: string;
}

const createHandle = (
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  dispatch: Dispatch<any>,
  props: Props,
  name: string,
  operation: string,
  isFormatValue: boolean,
) => {
  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  return (value: any): any =>
    dispatch(
      props.redux.slices.issueSlice.asyncActions.fetchMutationIssue(props.issueId, {
        name,
        operation,
        value: isFormatValue ? formatValue(value) : value,
      }),
    );
};

const ConnectedAttribute = connect<{}, {}, Props>(null, (dispatch, props) => {
  const name = props.backendUpdateKey || props.name;
  const isFormatValue = Boolean(props.backendUpdateKey);

  const log = (event: Event) => {
    rum.sendTimeMark('UX_attributeEventsOld', Date.now(), {
      ...event,
      time: Date.now(),
      attributeName: name,
      attributeComponent: props.component,
    });
  };

  const logHandler = (handler) => (value) => {
    const promise = handler(value);
    log(changeEvent());
    promise
      .then(() => {
        log(changeSuccessEvent());
      })
      .catch(() => {
        log(changeFailEvent());
      });
    return promise;
  };

  const handleChange = logHandler(createHandle(dispatch, props, name, 'change', isFormatValue));
  const handleAdd = logHandler(
    createHandle(dispatch, props, MAP_ADD[name] || name, 'add', isFormatValue),
  );
  const handleRemove = logHandler(
    createHandle(dispatch, props, MAP_REMOVE[name] || name, 'remove', isFormatValue),
  );
  const handleItemUpdate = logHandler(
    createHandle(dispatch, props, MAP_ADD[name] || name, 'update', isFormatValue),
  );
  const handleClear = logHandler(createHandle(dispatch, props, name, 'clear', false));

  return {
    adapter: issueAttributesAdapter,

    onEditingStart: () => {
      log(startEditingEvent());
    },

    onEditingStop: () => {
      log(stopEditingEvent());
    },
    onChange: handleChange,
    onAdd: handleAdd,
    onItemUpdate: handleItemUpdate,
    onRemove: handleRemove,
    onClear: handleClear,
  };
})(IssueAttributeWithContext);

export const IssueAttribute = withIssueProps(
  withFieldProps(withReduxContext<Props>(ConnectedAttribute)),
);
