import React from 'react';
import cx from 'classnames';
import Button from '@crm/components/dist/lego2/Button';
import Icon from 'lego/components/Icon';
import Access from 'utils/Access';
import IssueFieldLayout from '../IssueFieldLayout';
import css from './IssueField.module.css';
import Preview from './Preview';
import { FieldBaseProps } from './types';

interface State {
  edit: boolean;
}

export default abstract class IssueFieldBase extends React.Component<FieldBaseProps, State> {
  public static defaultProps = {
    previewComponent: 'PreviewValue',
  };

  public state = { edit: false };

  protected editRef = React.createRef<HTMLElement>();

  public componentDidUpdate(prevProps) {
    this.resetEditModeIfNeeded(prevProps);
  }

  private getIsEdit() {
    return this.state.edit && Access.isEdit(this.props.access);
  }

  protected toggle = () => {
    this.setState((prevState) => ({ edit: !prevState.edit }));
  };

  private resetEditModeIfNeeded(prevProps) {
    if (this.props.issueId !== prevProps.issueId) {
      this.setState({ edit: false });
    }
  }

  private renderIcon() {
    const { icon, iconWithValue = true, value } = this.props;

    // iconWithValue показываем иконку, когда выбрано значение
    if (!icon || (!iconWithValue && value)) {
      return null;
    }

    return <Icon {...icon} className={css.b__icon} />;
  }

  protected renderPreview() {
    const {
      editBtn,
      component,
      editByDblClick,
      value,
      multiline,
      wikiformatter,
      fieldProps = {},
      access,
      previewComponent,
      previewProps,
    } = this.props;

    let props: any = null;
    if (!editBtn && component && Access.isEdit(access)) {
      props = {
        ref: this.editRef,
        role: 'presentation',
      };

      if (editByDblClick) {
        props.onDoubleClick = this.toggle;
      } else {
        props.onClick = this.toggle;
      }
    }

    return (
      <span
        className={cx(
          css.b__label,
          { [css.b__label_oneline]: !multiline && !wikiformatter },
          { [css.b__label_multiline]: multiline },
          { [css.b__label_wikiformatter]: wikiformatter },
          css[`b__label_color_${fieldProps.color}`],
        )}
        {...props}
      >
        {this.renderIcon()}
        <Preview
          className={css.b__labelWrap}
          value={value}
          component={previewComponent}
          previewProps={previewProps}
        />
      </span>
    );
  }

  protected renderEditButton() {
    const { editBtn, component, access, editBtnTitle } = this.props;

    if (!(editBtn && component && Access.isEdit(access))) {
      return null;
    }

    return (
      <Button
        onClick={this.toggle}
        size="xs"
        view="pseudo"
        innerRef={this.editRef}
        className={css.b__editBtn}
        title={editBtnTitle}
        icon={(iconCls) => <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="edit" />}
      />
    );
  }

  protected renderRemoveButton() {
    const { value, onClear, remove, access, removeBtnTitle } = this.props;

    return (
      Access.isEdit(access) &&
      value != null &&
      remove && (
        <Button
          onClick={onClear}
          size="xs"
          view="pseudo"
          className={css.b__editBtn}
          title={removeBtnTitle}
          icon={(iconCls) => (
            <Icon className={iconCls} pack={Icon.PACKS.FONTAWESOME} icon="trash" />
          )}
        />
      )
    );
  }

  protected abstract renderField(isEditMode: boolean): React.ReactNode;

  protected abstract renderPreviewRow(isEditMode: boolean): React.ReactNode;

  public render() {
    const { width, className, size, color, weight, stopPropagation, label } = this.props;

    const edit = this.getIsEdit();

    return (
      <IssueFieldLayout
        label={label}
        width={width}
        className={className}
        size={size}
        color={color}
        weight={weight}
        stopPropagation={stopPropagation}
      >
        {this.renderField(edit)}
        {this.renderPreviewRow(edit)}
      </IssueFieldLayout>
    );
  }
}
