/* eslint-disable no-restricted-globals */

import React from 'react';
import { withProps } from 'recompose';
import Access from 'utils/Access';
import Button from '@crm/components/dist/lego2/Button';
import ButtonGroup2 from 'lego/components/ButtonGroup2';
import Tip from 'components/Tip';
import Tags from 'lego/components/Tags';
import { TimersSmallWidget } from 'modules/timer';
import { SupportChatWidgetWithLoadGuids } from 'components/SupportChatWidget';
import { connect, useSelector } from 'react-redux';
import { SupportChatIssue } from 'components/SupportChatIssue';
import { getCurrentUser } from 'entry/store/selectors';
import get from 'lodash/get';
import { InjectedReduxContextProps } from 'modules/issues/redux/types';
import { withOverlay } from 'components/Overlay';
import { useDynamicallyModalForm } from 'components/DynamicallyModalForm';
import Metrika from 'modules/metrika';
import DropdownMenu from 'lego/components/DropdownMenu';
import Icon from 'lego/components/Icon';
import { TagsWithAdd } from 'modules/tags';
import IssueDone from './components/IssueDone';
import IssueUnlink from './components/IssueUnlink';
import { allModals } from '../Forms';

const CONFIRM_MESSAGE = 'Подтвердите действие';

const getTagKey = (item) => item.id;

const actionItemsForMetrika = [
  'linkIssue',
  'createTicket1LineAction',
  'createTicket2LineAction',
  'DownloadFiles',
  'createStartrek',
  'linkStartrek',
];

export const IssueFieldCustomAdapters = {
  BUTTON2: withProps((props) => {
    const {
      onSubmit,
      issueId,
      access,
      name,
      alert,
      form,
      val,
      redirect,
      text,
      url,
      fieldProps = {},
      backendUpdateKey,
    } = props;

    const { icon, onClick, ...other } = fieldProps;

    const activityForm = useDynamicallyModalForm();

    return {
      onClick: (e) => {
        e?.stopPropagation?.();

        onClick?.();

        if (alert && !confirm(CONFIRM_MESSAGE)) {
          return null;
        }

        const options = { name: backendUpdateKey || name, alert, form, val, redirect, text, url };

        if (form) {
          allModals[form].open({
            onSubmit: (values) => onSubmit(values, options),
            issueId,
            actionName: val,
            activityForm,
            url,
          });
          return null;
        }

        return onSubmit(val, options);
      },
      disabled: !Access.isEdit(access),
      value: props.val,
      ...other,
    };
  }),
  BUTTON: withProps((props) => {
    const { onSubmit, issueId, access, alert, form, fieldProps = {} } = props;

    const { icon, ...other } = fieldProps;

    return {
      onClick: (e, value) => {
        e?.stopPropagation?.();

        if (alert && !confirm(CONFIRM_MESSAGE)) {
          return null;
        }

        if (form) {
          allModals[form].open({ onSubmit, issueId });
          return null;
        }

        return onSubmit(value);
      },
      disabled: !Access.isEdit(access),
      value: props.value,
      icon: icon ? (iconCls) => <Icon {...icon} className={iconCls} /> : undefined,
      ...other,
    };
  }),
  TOOLBAR: withProps((props) => {
    const { onSubmit, issueId, access, fieldProps = {}, isLoading = false } = props;

    const { icon, asButton, items } = fieldProps;

    const activityForm = useDynamicallyModalForm();
    const currentUser = useSelector(getCurrentUser);

    return {
      onClick: (e, value) => {
        e?.stopPropagation?.();

        if (asButton) {
          const obj = items.find((item) => item.val === value);

          if (obj.alert && !confirm(CONFIRM_MESSAGE)) {
            return null;
          }

          const options = { name: value, ...obj };

          if (actionItemsForMetrika.includes(obj.val)) {
            Metrika.reachGoal(`issueActions_${obj.val}`, {
              module: props?.redux?.name,
              userId: currentUser?.id,
              mainRole: currentUser?.mainRole,
            });
          }

          if (obj.form) {
            const modal = allModals[obj.form];

            if (modal) {
              modal.open({
                onSubmit: (values) => onSubmit(values, options),
                issueId,
                actionName: obj.val,
                activityForm,
                url: obj.url,
              });
            } else {
              throw new Error(`Форма для ${obj.form} не найдена`);
            }

            return null;
          }

          return onSubmit(value, options);
        }

        return onSubmit(value);
      },
      disabled: isLoading || !Access.isEdit(access),
      value: props.value,
      size: 'xs',
      children: icon ? <Icon {...icon} /> : null,
      onStateChange: (isVisible: boolean) => {
        if (isVisible && fieldProps.text === 'Действия') {
          Metrika.reachGoal(`issueActions_button`, {
            module: props?.redux?.name,
            userId: currentUser?.id,
            mainRole: currentUser?.mainRole,
          });
        }
      },
      ...fieldProps,
    };
  }),
  TAGS: withProps((props) => {
    const { onSubmit, access, fieldProps } = props;

    return {
      onAdd: (tagId) => onSubmit(tagId, { name: 'actionTagAdd' }),
      onRemove: (tagId) => onSubmit(tagId, { name: 'actionTagRemove' }),
      canEdit: Access.isEdit(access),
      tags: props.value,
      ...fieldProps,
    };
  }),
  FAST_TAGS: (WrappedComponent) => (props) => {
    const { value, fieldProps } = props;

    return <WrappedComponent getKey={getTagKey} items={value} {...fieldProps} />;
  },
  TIMERS: withProps((props) => {
    const { onSubmit, access, size } = props;

    const formatAddValues = (values) => {
      const sendData = { ...values };

      if (Array.isArray(sendData.users)) {
        sendData.userIds = sendData.users.map((item) => item.userId || item.id);
        delete sendData.users;
      }

      return sendData;
    };

    return {
      onAdd: (values) => onSubmit(formatAddValues(values), { name: 'editTimer' }),
      onRemove: (id) => {
        if (confirm(CONFIRM_MESSAGE)) {
          onSubmit(id, { name: 'deleteTimer' });
        }
      },
      disabled: !Access.isEdit(access),
      items: props.value,
      size,
    };
  }),
  TIP: withProps((props) => ({
    tip: props.value,
  })),
  SUPPORT_CHAT: withProps((props) => {
    const meta = {
      issueId: props.issueId,
      accountId: get(props, 'currentUser.id'),
      queueId: get(props, 'currentIssue.data.queue.id'),
      categoryId: get(props, 'currentIssue.data.category.id'),
      comType: get(props, 'currentIssue.data.communicationTypeId.name'),
      ticketLink: `https://crm.yandex-team.ru/space/index.html#/cases/${props.issueId}`,
    };
    const validator = () => {
      if (!props.currentIssue.data.category) {
        return 'Чтобы начать диалог с Оператором 1.5 линии, задайте категорию в тикете';
      }
      if (!props.currentIssue.data.owner) {
        return 'Чтобы начать диалог с Оператором 1.5 линии, нужно быть Исполнителем тикета';
      }
      if (props.currentIssue.data.owner.id !== props.currentUser.id) {
        return 'Чтобы начать диалог с Оператором 1.5 линии, нужно быть Исполнителем тикета';
      }
      return true;
    };
    return {
      comType: get(props, 'currentIssue.data.communicationTypeId.name'),
      access: props.access,
      meta,
      validators: [validator],
      view: props.fieldProps?.view,
    };
  }),
};

export default {
  Button: IssueFieldCustomAdapters.BUTTON(Button),
  Button2: IssueFieldCustomAdapters.BUTTON2(Button),
  ButtonGroup: IssueFieldCustomAdapters.TOOLBAR(
    withOverlay('isLoading')(
      ButtonGroup2,
      { zIndex: 10, size: 'xxs' },
      { maxWidth: '100%', display: 'inline-flex', flex: 1 },
    ),
  ),
  DropdownMenu: IssueFieldCustomAdapters.TOOLBAR(DropdownMenu),
  IssueDone: IssueFieldCustomAdapters.BUTTON(IssueDone),
  IssueUnlink: IssueFieldCustomAdapters.BUTTON(IssueUnlink),
  IssueTags: IssueFieldCustomAdapters.TAGS(TagsWithAdd),
  IssueFastTags: IssueFieldCustomAdapters.FAST_TAGS(Tags),
  Timers: IssueFieldCustomAdapters.TIMERS(TimersSmallWidget),
  Tip: IssueFieldCustomAdapters.TIP(Tip),
  SupportChat: connect<{}, {}, InjectedReduxContextProps & { issueId: string }>(
    (state, ownProps) => ({
      currentIssue: ownProps.redux.selectors.storage.issues.getItem(state, ownProps.issueId),
      currentUser: getCurrentUser(state),
    }),
  )(IssueFieldCustomAdapters.SUPPORT_CHAT(SupportChatWidgetWithLoadGuids)),
  SupportChatIssue: IssueFieldCustomAdapters.SUPPORT_CHAT(SupportChatIssue),
};
