import React, { useCallback } from 'react';
import cx from 'classnames';
import TextInput from 'lego/final-form/TextInput';
import WFInput from 'lego/final-form/WFInput';
import SuggestStaffInline2 from 'lego/final-form/SuggestStaffInline2';
import DateTimePicker from 'lego/final-form/DateTimePicker';
import { Field, Form } from 'react-final-form';
import Button from '@crm/components/dist/lego2/Button';
import Popup from '@crm/components/dist/lego2/Popup';
import Prompt from 'components/Prompt';
import Icon from 'lego/components/Icon';
import Toolbar from 'components/Toolbar';
import css from './IssueFieldEditForm.module.css';

const InputsMap = {
  TextInput,
  WFInput,
  Date: DateTimePicker,
  SuggestStaffInline2,
};

const INJECT_ACTIONS_FIELDS = {
  WFInput: true,
};

const config = {
  inject: ({ onCancel, submitting }) => [
    <Button type="submit" view="action" key="save" disabled={submitting} progress={submitting}>
      Сохранить
    </Button>,
    <Button onClick={onCancel} view="pseudo" key="cancel" disabled={submitting}>
      Отменить
    </Button>,
  ],
  inline: ({ onCancel, submitting }) => (
    <React.Fragment>
      <Button
        type="submit"
        view="pseudo"
        disabled={submitting}
        progress={submitting}
        icon={(iconCls) => <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="done" />}
      />
      <Button
        onClick={onCancel}
        view="pseudo"
        disabled={submitting}
        icon={(iconCls) => <Icon className={iconCls} pack={Icon.PACKS.MATERIAL} icon="block" />}
      />
    </React.Fragment>
  ),
  popup: ({ onCancel, onClear, submitting }) => (
    <Toolbar className={css.b__footer}>
      <Button type="submit" view="action" disabled={submitting} progress={submitting}>
        Сохранить
      </Button>
      <Button view="pseudo" onClick={onCancel} disabled={submitting}>
        Отменить
      </Button>
      <Button view="danger" onClick={onClear} className={css.b__clear} disabled={submitting}>
        Очистить
      </Button>
    </Toolbar>
  ),
};

const InlineForm = (props) => {
  const { type, value, onSubmit, onClear, onCancel, fieldProps, component } = props;

  const handleSubmit = useCallback(
    (values) => {
      onSubmit(values.value).then((data) => {
        onCancel();

        return data;
      });
    },
    [onSubmit, onCancel],
  );

  const handleClear = useCallback(() => {
    onClear().then((data) => {
      onCancel();

      return data;
    });
  }, [onClear, onCancel]);

  return (
    <Form onSubmit={handleSubmit} initialValues={{ value }}>
      {(formApi) => (
        <form className={cx(css.b, css[`b_type_${type}`])} onSubmit={formApi.handleSubmit}>
          <Prompt when={type === 'inline' && formApi.dirty} preventCloseBrowser />
          <Field
            component={InputsMap[component] || component}
            name="value"
            className={css.b__field}
            {...fieldProps}
            actions={INJECT_ACTIONS_FIELDS[component] ? config.inject(props) : undefined}
            autoFocus
          />
          {!INJECT_ACTIONS_FIELDS[component] &&
            config[type]({ submitting: formApi.submitting, onCancel, onClear: handleClear })}
        </form>
      )}
    </Form>
  );
};

InlineForm.defaultProps = {
  type: 'inline',
};

const PopupForm = (props) => (
  <Popup
    visible
    target="anchor"
    anchor={props.editRef}
    padding="xs"
    onOutsideClick={props.onCancel}
  >
    <InlineForm {...props} type="popup" />
  </Popup>
);

export { InlineForm, PopupForm };
