import React from 'react';
import { Field, Form } from 'react-final-form';
import { connect } from 'react-redux';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import { selectors } from 'modules/initial';
import FieldRadioButton from 'lego/final-form/RadioButton';
import css from './Filters.module.css';

interface Values {
  state: string;
  panel: string;
}

interface ConnectedState {
  initialValues: Values;
  isTeamlead: boolean;
}

interface ConnectedDispatch {
  onSubmit: (data: Values) => void;
}

interface OwnProps {
  className?: string;
}

type Props = ConnectedState & ConnectedDispatch & InjectedReduxContextProps & OwnProps;

const Filters: React.FC<Props> = (props) => {
  const { isTeamlead, initialValues, className } = props;
  const panelOptions = [
    { value: 'My', children: 'Мои' },
    { value: 'Author', children: 'Делегированы' },
    { value: 'Follower', children: 'Наблюдаю' },
  ];

  if (isTeamlead) {
    panelOptions.push({ value: 'MyGroup', children: 'Моя группа' });
  }

  const stateOptions = [
    { value: 'None', children: 'Все' },
    { value: 'Open', children: 'Открытые' },
    { value: 'Closed', children: 'Закрытые' },
  ];

  return (
    // @ts-ignore
    <Form initialValues={initialValues} onSubmit={() => {}} validate={props.onSubmit}>
      {(formApi) => (
        <form className={className} onSubmit={formApi.handleSubmit}>
          <Field
            name="state"
            component={FieldRadioButton}
            options={stateOptions}
            className={css.stateRadioButton}
          />
          <Field name="panel" component={FieldRadioButton} options={panelOptions} />
        </form>
      )}
    </Form>
  );
};

export default withReduxContext<OwnProps>(
  connect<ConnectedState, ConnectedDispatch, InjectedReduxContextProps>(
    (state, props) => ({
      initialValues: props.redux.selectors.getFastFilter(state),
      isTeamlead: selectors.isTeamleadOrGrouplead(state),
    }),
    (dispatch, props) => ({
      onSubmit: (data) =>
        dispatch(props.redux.slices.filterSlice.actions.setWithRefresh({ type: 'fast', data })),
    }),
  )(Filters),
);
