import React from 'react';
import { connect } from 'react-redux';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import { VirtualList, WithNavigationVirtualList } from 'components/InfiniteList';
import debounce from 'lodash/debounce';
import IssueItem from '../Issue/IssueItem';
import storagePathCreator from '../../utils/storagePathCreator';
import { Node, ModuleName } from '../../types';
import css from './Issues.module.css';
import { IssueMeasure } from './IssueMeasure';
import basicScheme, { opportunitiesScheme } from '../schemes/schemeMainListItem';

interface ConnectedState {
  selected: number | string;
  node: Node;
  isFetch: boolean;
  isLoadingFirstPage?: boolean;
}

interface ConnectedDispatch {
  more: (offset: number) => void;
}

interface ConnectedProps {
  className?: string;
  id: number;
  path: string;
}

interface State {
  id: string | number;
}

type Props = ConnectedState & ConnectedProps & ConnectedDispatch & InjectedReduxContextProps;

class IssueList extends React.Component<Props, State> {
  private issueMeasure: IssueMeasure;

  private listRef = React.createRef<VirtualList>();

  private redirectDelay: number = 350;

  private renderedItemIds: number[] = [];

  private handleChange = debounce((id) => {
    this.props.redux.slices.issueSlice.goToIssue({
      issueId: id,
      blank: false,
    });
  }, this.redirectDelay);

  public constructor(props) {
    super(props);

    this.issueMeasure = new IssueMeasure();
    this.issueMeasure.setRedux(props.redux);
    this.issueMeasure.setListRef(this.listRef);
  }

  private handleLoad = () => {
    const { node, more } = this.props;

    const offset = ((node || {}).items || []).length;

    more(offset);
  };

  private renderIssue = ({ index, style, data }) => {
    const itemId = this.getItemKey(index);
    const selected = data[index];
    return (
      <div style={style} className={css.b__item}>
        <IssueItem
          className={css.b__issueItem}
          scheme={this.getItemScheme()}
          id={itemId}
          selected={selected}
          node={this.props.node} // FIXME CRM-16938: TEMP FOR LOG
        />
      </div>
    );
  };

  private getItemScheme = () => {
    const { redux } = this.props;
    if (redux.name === ModuleName.Opportunities) {
      return opportunitiesScheme;
    }

    return basicScheme;
  };

  private getItemKey = (index) => {
    return this.renderedItemIds[index];
  };

  public render() {
    const { node, isFetch, className, isLoadingFirstPage } = this.props;
    const itemCount = ((node || {}).items || []).length;

    if (!node) {
      return null;
    }

    this.renderedItemIds = node.items || [];
    this.issueMeasure.setItemIds(this.renderedItemIds);

    return (
      <span style={{display: 'contents'}} data-testid="issues-list/list">
        <WithNavigationVirtualList
          listRef={this.listRef}
          isEof={node.eof}
          isLoading={isFetch}
          isLoadingFirstPage={isLoadingFirstPage}
          onLoad={this.handleLoad}
          itemCount={itemCount}
          itemKey={this.getItemKey}
          itemSize={this.issueMeasure.getItemSize}
          itemIds={this.renderedItemIds}
          selectedId={this.props.selected}
          className={className}
          onChange={this.handleChange}
        >
          {this.renderIssue}
        </WithNavigationVirtualList>
      </span>
    );
  }
}

export default withReduxContext<ConnectedProps>(
  connect<ConnectedState, ConnectedDispatch, ConnectedProps & InjectedReduxContextProps>(
    (state, { id, path, redux }) => {
      return {
        node: redux.selectors.storage.nodes.getItem(state, path, id),
        selected: redux.selectors.getMainId(state),
        ...redux.selectors.meta.nodes(state, storagePathCreator.merge(path, id)),
      };
    },
    (dispatch, props) => ({
      more: (offset) => {
        const { id, path } = props;

        dispatch(
          props.redux.slices.nodesSlice.asyncActions.get({
            parentId: id,
            offset,
            path,
            nodeId: storagePathCreator.merge(path, id),
          }),
        );
      },
    }),
  )(IssueList),
);
