import React from 'react';
import store from 'store';
import { Redux } from 'modules/issues/redux';
import { IssueObject } from 'modules/issues/types';
import Access from 'utils/Access';
import { VirtualList } from 'components/InfiniteList';

export class IssueMeasure {
  private listRef?: React.RefObject<VirtualList>;

  private heightMap: WeakMap<IssueObject, number>;

  private redux: Redux | null;

  private itemIds: number[];

  private unsubscribe: () => void;

  // eslint-disable-next-line
  private currentIssueState: any;

  public constructor() {
    this.heightMap = new Map();
    this.unsubscribe = store.subscribe(this.handleStateUpdate);
  }

  public setListRef(listRef: React.RefObject<VirtualList>) {
    this.listRef = listRef;
  }

  public setItemIds(itemIds: number[]) {
    this.itemIds = itemIds;
  }

  public setRedux(redux: Redux): void {
    this.redux = redux;
  }

  private handleStateUpdate = () => {
    if (!this.redux || !(this.listRef && this.listRef.current)) {
      return;
    }

    const previousIssueState = this.currentIssueState;
    this.currentIssueState = this.redux.selectors.storage.issues.getAll(store.getState());

    if (previousIssueState !== this.currentIssueState) {
      this.listRef.current.resetAfterIndex(0);
    }
  };

  public getItemSize = (index: number): number => {
    const { redux } = this;

    let size = 50; // base size

    if (!redux) {
      return size;
    }

    const issueObj: IssueObject | undefined = redux.selectors.storage.issues.getItem(
      store.getState(),
      this.itemIds[index],
    );

    if (!issueObj) {
      return size;
    }

    const storeSize = this.heightMap.get(issueObj);
    if (storeSize) {
      return storeSize;
    }

    const hasParent = Boolean(
      Access.isRead(issueObj.props.parentIssue.access) && issueObj.data.parentIssue,
    );

    const hasTags = Boolean(Access.isRead(issueObj.props.tags.access) && issueObj.data.tags.length);
    const hasCategory = Boolean(Access.isRead(issueObj.props.category.access));
    const hasOpportunityAmount = Boolean(
      Access.isRead(issueObj.props.opportunityCurrentAmount.access),
    );

    if (hasParent) {
      size += 20;
    }

    if (hasCategory) {
      size += 20;
    }

    if (hasTags) {
      size += 25;
    }

    if (hasOpportunityAmount) {
      size += 20;
    }

    this.heightMap.set(issueObj, size);

    return size;
  };
}
