import React, { useState } from 'react';
import cx from 'classnames';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import Button from '@crm/components/dist/lego2/Button';
import Access from 'utils/Access';
import { connect } from 'react-redux';
import Filters from './Filters';
import Search from './Search';
import Sort from './Sort';
import CreateNewIssue from '../CreateNewIssue';
import css from './Issues.module.css';
import IssueList from './IssueList';
import { modal as createFormModal, CreateFormValues, CasesFormValues } from '../Forms/CreateForm';
import {
  CreateCaseIssueBackendData,
  CreateOpportunityBackendData,
  ModuleName,
  IssueType,
} from '../../types';

const isValuesForCase = (values: CreateFormValues): values is CasesFormValues => {
  return (values as CasesFormValues).category != null;
};

const createIssueFrontendToBackend = (
  values: CreateFormValues,
): CreateCaseIssueBackendData | CreateOpportunityBackendData => {
  if (isValuesForCase(values)) {
    const result: CreateCaseIssueBackendData = {
      name: values.name.name,
      queueId: values.queue.id,
      accountId: values.account.id,
      categoryId: values.category.id,
      workflowId: values.workflowId,
      stateId: values.stateId,
      count: values.count,
    };

    if (values.owner) {
      result.ownerId = values.owner.id;
    }

    return result;
  }

  const result: CreateOpportunityBackendData = {
    name: values.name,
    opportunitySourceId: Number(values.opportunitySource),
    accountId: values.account.id,
  };

  if (values.owner) {
    result.ownerId = values.owner.id;
  }

  return result;
};

interface TakeNextResponse {
  data?: {
    issue: {
      id: number;
      typeId: number;
    };
  };
  storage: {};
  message?: [];
}

interface ConnectedState {
  hasFilters: boolean;
  hasFastCreateIssue: boolean;
  hasNextTicketButton: boolean;
  hasCreateButton: boolean;
  searchFieldsAccess: { [key: string]: number };
  takeNextAvailableCount?: number;
}

interface ConnectedDispatch {
  onTakeNextClick: () => Promise<TakeNextResponse>;
  onCreateNew: (data: CreateFormValues) => void;
}

type Props = ConnectedState & ConnectedDispatch & InjectedReduxContextProps;

const Issues: React.FC<Props> = (props) => {
  const {
    hasFilters,
    hasFastCreateIssue,
    searchFieldsAccess,
    hasNextTicketButton,
    hasCreateButton,
    onTakeNextClick,
    onCreateNew,
    takeNextAvailableCount,
    redux,
  } = props;

  const [isLoading, setLoading] = useState(false);

  const handleTakeNextClick = () => {
    setLoading(true);
    onTakeNextClick()
      .then((response) => {
        if (response.data) {
          const { issue } = response.data;

          redux.slices.issueSlice.goToIssue({
            moduleName: ModuleName[IssueType[issue.typeId]],
            issueId: issue.id,
            blank: false,
          });
        }
      })
      .finally(() => {
        setLoading(false);
      });
  };

  return (
    <div data-testid="issues-list" className={css.b}>
      {(hasNextTicketButton || hasCreateButton) && (
        <div data-testid="issues-list/buttons" className={css.b__toolbar}>
          {hasNextTicketButton && (
            <Button
              view="action"
              pin="brick-brick"
              size="m"
              progress={isLoading}
              className={css.b__takeNextButton}
              onClick={handleTakeNextClick}
              title={
                takeNextAvailableCount !== undefined
                  ? `Доступных тикетов в очереди ${takeNextAvailableCount}`
                  : undefined
              }
            >
              Взять следующий
              {takeNextAvailableCount !== undefined && ` (${takeNextAvailableCount})`}
            </Button>
          )}
          {hasCreateButton && (
            <Button
              className={cx(css.b__createButton, {
                [css.b__createButton_width_max]: !hasNextTicketButton,
              })}
              view="action"
              pin="brick-brick"
              size="m"
              onClick={() =>
                createFormModal.open({ onSubmit: onCreateNew, moduleName: redux.name })
              }
            >
              +
            </Button>
          )}
        </div>
      )}
      <div data-testid="issues-list/search" className={css.b__panel}>
        {hasFilters && <Filters className={css.b__filters} />}
        <Search fieldsAccess={searchFieldsAccess} />
        {hasFastCreateIssue && <CreateNewIssue className={css.b__new} />}
        <Sort />
      </div>
      <IssueList className={css.b__list} id={0} path="issueList" />
    </div>
  );
};

export default withReduxContext(
  connect<ConnectedState, ConnectedDispatch, InjectedReduxContextProps>(
    (state, props) => {
      const moduleSettings = props.redux.selectors.moduleSettings(state);

      return {
        hasFilters: Access.isEdit(moduleSettings.filters),
        hasFastCreateIssue: Access.isEdit(moduleSettings.create.inline),
        hasNextTicketButton: Access.isEdit(moduleSettings.takeNext),
        hasCreateButton: Access.isEdit(moduleSettings.create.modal),
        searchFieldsAccess: moduleSettings.search,
        takeNextAvailableCount: props.redux.selectors.storage.getTakeNextAvailableCount(state),
      };
    },
    (dispatch, props) => ({
      onTakeNextClick: () =>
        dispatch(props.redux.slices.moduleSlice.asyncActions.takeNextTicket({})),
      onCreateNew: (data: CreateFormValues) =>
        dispatch(
          props.redux.slices.issueSlice.asyncActions.createIssue(
            createIssueFrontendToBackend(data),
          ),
        ),
    }),
  )(Issues),
);
