import * as React from 'react';
import { connect } from 'react-redux';
import {
  Provider as Categorization2Provider,
  Modal as Categorization2Modal,
} from 'modules/categorization2';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import { Subscription } from 'rxjs';
import { ModuleTitle } from 'modules/issues/types';
import { OwnProps as IssueProps } from 'modules/issues/components/Issue/Issue';
import { modal } from 'modules/requestForm2';
import TitleChain from 'modules/heading';
import { subject, store } from '../CategorizationSetup';
import Issue, { IssueName } from '../Issue';

interface StateProps {
  isTest: boolean;
}

interface DispatchProps {
  mount: () => void;
  unmount: () => void;
}

interface OwnProps extends IssueProps {
  redirect?: () => void;
}

interface IssueMainProps extends StateProps, DispatchProps, OwnProps, InjectedReduxContextProps {}

class IssueMain extends React.Component<IssueMainProps> {
  private sub: Subscription;

  public componentDidMount(): void {
    this.sub = subject.subscribe(({ event, payload }) => {
      if (window.Ya && window.Ya.Rum) {
        window.Ya.Rum.sendTimeMark('categorizationV3', window.Ya.Rum.getTime(), false, {
          '-env': this.props.isTest ? 'testing' : 'production',
          '-additional': encodeURIComponent(
            JSON.stringify({
              event,
              eType: payload.type,
              eId: payload.id,
              userId: window.USER_ID,
              categoryId: payload.categoryId,
              block: payload.block,
              order: payload.order,
              resultsCount: payload.resultsCount,
              text: payload.text,
              hasTip: payload.hasTip,
              commentId: payload.commentId,
              categoryIdHasAlert: payload.categoryIdHasAlert,
              hasAlert: payload.hasAlert,
            }),
          ),
        });
      }
    });
    this.props.mount();
  }

  public componentDidUpdate(prevProps): void {
    if (this.props.id !== prevProps.id) {
      modal.destroy();
      this.props.mount();
    }
  }

  public componentWillUnmount(): void {
    this.props.unmount();
    this.sub.unsubscribe();
    modal.destroy();
  }

  public render() {
    const { id, redux, ...other } = this.props;

    return (
      <Categorization2Provider store={store}>
        <React.Fragment>
          {id && (
            <React.Fragment>
              <TitleChain>{ModuleTitle[redux.name]}</TitleChain>
              <TitleChain>{id}</TitleChain>
              <IssueName id={id} component={TitleChain} />
            </React.Fragment>
          )}
          <Issue id={id} {...other} />
        </React.Fragment>
        <Categorization2Modal key={id} />
      </Categorization2Provider>
    );
  }
}

export default withReduxContext<OwnProps>(
  connect<
    StateProps,
    DispatchProps,
    OwnProps & InjectedReduxContextProps,
    {
      root: {
        info: {
          testMode: boolean;
        };
      };
    }
  >(
    (state) => ({
      isTest: Boolean(state.root.info.testMode),
    }),
    (dispatch, props) => ({
      mount: () => dispatch(props.redux.slices.mainSlice.actions.set(props.id)),
      unmount: () => dispatch(props.redux.slices.mainSlice.actions.clear()),
    }),
  )(IssueMain),
);
