import React from 'react';
import UrlFilterBinder from 'components/UrlFilterBinder';
import { connect } from 'react-redux';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import * as initial from 'modules/initial';
import { DEFAULT_FILTER } from '../constants';
import { ModuleName } from '../types';

const STATE_VALUES = ['None', 'Open', 'Closed'];
const PANEL_VALUES = ['My', 'Author', 'Follower', 'MyGroup'];

interface DefaultValues {
  filterId?: number;
  state?: string;
  panel?: string;
}

const createParsers = (defaultValues: DefaultValues = {}) => ({
  filterId: (value = defaultValues.filterId) => Number(value),
  state: (value) => {
    if (STATE_VALUES.includes(value)) {
      return value;
    }

    return defaultValues.state;
  },
  panel: (value) => {
    if (
      !PANEL_VALUES.includes(value) ||
      (value === 'MyGroup' &&
        !initial.selectors.isTeamleadOrGrouplead(window.reduxStore.getState()))
    ) {
      return defaultValues.panel;
    }

    return value;
  },
  search: (value) => {
    try {
      return JSON.parse(value);
    } catch (e) {
      /* empty catch */
    }

    return null;
  },
});

const formattors = {
  search: (value) => JSON.stringify(value),
};

const issueFields = ['filterId', 'state', 'panel', 'search'];
const ticketFields = ['filterId', 'search'];
const opportunitiesFields = ['filterId', 'search'];

const createProccessor = (processors, fields) => (data = {}) =>
  fields.reduce((ac, field) => {
    let proccessedValue;

    if (typeof processors[field] === 'function') {
      proccessedValue = processors[field](data[field]);
    } else {
      proccessedValue = data[field];
    }

    if (proccessedValue) {
      // eslint-disable-next-line no-param-reassign
      ac[field] = proccessedValue;
    }

    return ac;
  }, {});

const adapters = {
  [ModuleName.Issue]: {
    parse: createProccessor(createParsers(DEFAULT_FILTER[ModuleName.Issue]), issueFields),
    format: createProccessor(formattors, issueFields),
  },
  [ModuleName.Ticket]: {
    parse: createProccessor(createParsers(DEFAULT_FILTER[ModuleName.Ticket]), ticketFields),
    format: createProccessor(formattors, ticketFields),
  },
  [ModuleName.Opportunities]: {
    parse: createProccessor(
      createParsers(DEFAULT_FILTER[ModuleName.Opportunities]),
      opportunitiesFields,
    ),
    format: createProccessor(formattors, opportunitiesFields),
  },
};

interface Filter {
  // eslint-disable-next-line
  [key: string]: any;
}

interface ConnectedState {
  filter: Filter;
}

interface ConnectedDispatch {
  onFilterChange: (filter: Filter) => void;
}

interface OwnProps {
  module: ModuleName;
}

type Props = ConnectedState & ConnectedDispatch & OwnProps;

const UrlFilterBinderWithProps: React.FC<Props> = (props) => (
  <UrlFilterBinder {...props} {...adapters[props.module]} />
);

export default withReduxContext<OwnProps>(
  connect<ConnectedState, ConnectedDispatch, OwnProps & InjectedReduxContextProps>(
    (state, props) => ({
      filter: props.redux.selectors.getCurrentFilterDataForUrl(state),
    }),
    (dispatch, props) => ({
      onFilterChange: (data) => {
        if (data && data.search) {
          dispatch(
            props.redux.slices.filterSlice.actions.setWithRefresh({
              type: 'search',
              data: data.search,
            }),
          );
        } else {
          dispatch(props.redux.slices.filterSlice.actions.setWithRefresh({ type: 'fast', data }));
        }
      },
    }),
  )(UrlFilterBinderWithProps),
);
