import * as React from 'react';
import Promise from 'bluebird';
import { Position, Tone } from '@crm/components/dist/Panel';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import { AdaptiveIntervalByTabVisibility, Interval } from 'services/AdaptiveInterval';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import 'assets/ys-font/fonts.css';
import Resizable from 'modules/resizable';
import { config } from 'services/Config';
import Panel from 'modules/panel';
import IssueMainRouter from './IssueMainRouter';
import IssueMainList from './IssueMainList';
import { IssueModuleRefresh } from './IssueModuleRefresh';
import Filters from './Filters';
import IssueUrlFilterBinder from './IssueUrlFilterBinder';
import css from './Page.module.css';
import { PageProps, ConnectedDispatch } from './Page.types';

class Page extends React.Component<PageProps> {
  private interval: Interval;

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  private promise: Promise<any> | null = null;

  private controller = new AbortController();

  public componentDidMount() {
    this.props.get(
      {
        nodeId: 'issueList.map.0',
        path: 'issueList',
      },
      this.controller.signal,
    );
    this.props.getFilters(
      {
        nodeId: 'issueFilters.map.0',
        path: 'issueFilters',
      },
      this.controller.signal,
    );
    this.interval = new AdaptiveIntervalByTabVisibility(() => {
      this.refresh();
    }, config.value.config.intervals.issuesRefresh);
  }

  public componentWillUnmount() {
    this.interval.destroy();
    this.cancel();
    this.controller.abort();
    this.props.destroy();
  }

  private cancel() {
    if (this.promise) {
      this.promise.cancel();
    }
  }

  private refresh = () => {
    this.cancel();
    this.promise = this.props.refresh({});
  };

  public render() {
    return (
      <div data-testid="page-issues" className={css.page}>
        <Panel
          name="issue/filters"
          position={Position.Left}
          defaultPin
          hasPinButton={!config.value.features.newDesignFilters}
          defaultPanelId="filters"
          defaultSize={{
            width: 200,
          }}
          minWidth={100}
          maxWidth={300}
          tone={Tone.Neowhite}
          panels={[
            {
              id: 'filters',
              title: config.value.features.newDesignFilters ? 'Папки' : 'Фильтры',
              padding: false,
              overflow: false,
              content: <Filters />,
            },
          ]}
        />
        <Resizable
          enable="right"
          wrap
          name="issue/list"
          defaultSize={{
            width: 350,
          }}
          minWidth={270}
          maxWidth="35%"
        >
          <IssueMainList />
          <IssueModuleRefresh refresh={this.refresh} />
        </Resizable>
        <IssueMainRouter module={this.props.redux.name} />
        <IssueUrlFilterBinder module={this.props.redux.name} />
      </div>
    );
  }
}

export default withReduxContext(
  connect<{}, ConnectedDispatch, InjectedReduxContextProps>(null, (dispatch, props) =>
    bindActionCreators(
      {
        get: props.redux.slices.nodesSlice.asyncActions.get,
        getFilters: props.redux.slices.nodesSlice.asyncActions.getFilters,
        refresh: props.redux.slices.moduleSlice.asyncActions.refresh,
        destroy: props.redux.slices.moduleSlice.actions.destroy,
      },
      dispatch,
    ),
  )(Page),
);
