import React, { PureComponent } from 'react';
import cx from 'classnames';
import { Audio } from 'components/Audio';
import { config } from 'services/Config';
import CopyButton from 'lego/components/CopyButton';
import Lego2Icon from '@crm/components/dist/lego2/Icon';
import { audioRecordSrc } from 'utils/audioRecordSrc';
import { audioMetaSrc } from 'utils/audioMetaSrc';
import { AudioSource } from './AudioSource';
import { CallInfoProps } from './CallInfo.types';
import CallPreview from './CallPreview';
import css from './CallInfo.module.css';
import getFieldsList from './utils/getFieldsList';
import FieldPlaceholder from './FieldPlaceholder';

export default class CallInfo extends PureComponent<CallInfoProps> {
  private getRecordLink = (call) => {
    const isTimelineV2 = config.value.features.issueTimelineV2;

    if (!call.id) {
      return '';
    }

    const audioProps = {
      key: call.sync,
      src: audioRecordSrc(call.id),
      metaSrc: audioMetaSrc(call.id),
    };

    return (
      <div className={css.b__audioBlock}>
        {isTimelineV2 ? <AudioSource {...audioProps} /> : <Audio {...audioProps} />}
        {call.recordId && (
          <CopyButton
            iconProvider={(cls) => <Lego2Icon className={cls} svg="copy2" svgSize="ms" />}
            copy={`https://yacalls.yandex-team.ru/recordings/${call.recordId}`}
            size="xs"
            view="clear"
            tooltipText="Ссылка скопирована"
            withTooltip
            tooltipCloseDelay={1500}
          />
        )}
      </div>
    );
  };

  render() {
    const { call, preview, view, categorization, activityData = {} } = this.props;

    const list = getFieldsList(view, call, this.getRecordLink, categorization, activityData);
    const isInAccountHistory = view === 'accountHistory' || view === 'accountHistoryPersonal';

    return (
      <div>
        {call.callStatus.id === 2 && preview ? (
          <CallPreview call={call} />
        ) : (
          <ul data-testid="fields-list" className={css.b}>
            {list.map(([key, value, fieldName]) => (
              <li key={key} className={css.b__item}>
                <span
                  className={cx(css.b__fieldName, {
                    [css.b__fieldName_view_small]: isInAccountHistory,
                  })}
                >
                  {key}:
                </span>
                {fieldName && value === undefined ? (
                  <FieldPlaceholder fieldName={String(fieldName)} />
                ) : (
                  value
                )}
              </li>
            ))}
          </ul>
        )}
      </div>
    );
  }
}
