import React from 'react';
import get from 'lodash/get';
import { IssueTags } from 'modules/issues/components/IssueTags';
import { CategoriesPreviewFromValue } from 'modules/categorization/components/CategoriesPreview';
import { UpsaleFactorsComponent } from 'components/UpsaleFactors';
import { TagListWithAdapter } from 'components/TagList';
import CONSTANTS from '../CallInfo.constants';
import { sec2time } from '../../../../utils/sec2time';
import { ActivityTags } from '../ActivityTags';
import { ListView, GetRecordLink, FieldsList, Call, Categorization } from '../CallInfo.types';
import { CallActivityData } from '../../Call.types';

const kikDataList = (call) => {
  const isHidden = Boolean(call.kik && call.kik.hidden);
  if (isHidden) {
    return [
      [CONSTANTS.name, undefined, 'kikHidden'],
      [CONSTANTS.phone, undefined, 'kikHidden'],
    ];
  }
  return [
    [CONSTANTS.name, get(call, 'clientInfo.name') || get(call, 'kik.name'), 'kikName'],
    [CONSTANTS.phone, get(call, 'clientInfo.phone') || get(call, 'kik.phoneE164'), 'kikPhone'],
  ];
};

const getFactorsRow = (call: Call) => {
  return [
    CONSTANTS.upsaleFactors,
    React.createElement(UpsaleFactorsComponent, { factors: call.upsaleFactors }),
  ];
};

const getActivityTagsRow = (activityData: CallActivityData) => {
  return [CONSTANTS.tags, React.createElement(ActivityTags, activityData)];
};

export const getRegularList = (
  call,
  _getRecordLink,
  _categorization,
  activityData?,
): FieldsList => {
  const list = [
    ...kikDataList(call),
    [CONSTANTS.operatorPhone, get(call, 'operatorPhoneNumber')],
    [CONSTANTS.status, get(call, 'callStatus.name')],
    getFactorsRow(call),
  ];
  if (activityData?.showActivityTags) {
    list.push(getActivityTagsRow(activityData));
  }
  return list;
};

export const getFullList = (call, getRecordLink, _categorization, activityData?): FieldsList => {
  const list = [
    ...kikDataList(call),
    [CONSTANTS.operatorPhone, get(call, 'operatorPhoneNumber')],
    [CONSTANTS.status, get(call, 'callStatus.name')],
    [CONSTANTS.reason, get(call, 'endReason.name')],
    [CONSTANTS.duration, sec2time(call.duration)],
    [CONSTANTS.record, getRecordLink(call)],
    [CONSTANTS.callId, get(call, 'callId')],
    getFactorsRow(call),
  ];
  if (activityData?.showActivityTags) {
    list.push(getActivityTagsRow(activityData));
  }
  return list;
};

export const getListForAccountHistory = (call, getRecordLink, categorization?): FieldsList => {
  return [
    ...kikDataList(call),
    [
      CONSTANTS.categories,
      React.createElement(CategoriesPreviewFromValue, { value: categorization }),
    ],
    [CONSTANTS.status, get(call, 'callStatus.name')],
    [CONSTANTS.reason, get(call, 'endReason.name')],
    [CONSTANTS.duration, sec2time(call.duration)],
    [CONSTANTS.record, getRecordLink(call)],
    getFactorsRow(call),
  ];
};

export const getListForAccountHistoryPersonal = (
  call,
  getRecordLink,
  categorization?,
): FieldsList => {
  return [
    ...kikDataList(call),
    [
      CONSTANTS.categories,
      React.createElement(CategoriesPreviewFromValue, { value: categorization }),
    ],
    [CONSTANTS.status, get(call, 'callStatus.name')],
    [CONSTANTS.reason, get(call, 'endReason.name')],
    [CONSTANTS.duration, sec2time(call.duration)],
    [
      CONSTANTS.record,
      getRecordLink({
        id: call.eid,
        sync: call.sync,
      }),
    ],
    [CONSTANTS.opportunities, React.createElement(IssueTags, { issues: call.issues })],
    [
      CONSTANTS.tags,
      React.createElement(TagListWithAdapter, { tags: call.tags, placeholder: true }),
    ],
    [CONSTANTS.comment, get(call, 'comment')],
    [CONSTANTS.callId, get(call, 'callId')],
  ];
};

const gettersMap = {
  regular: getRegularList,
  full: getFullList,
  accountHistory: getListForAccountHistory,
  accountHistoryPersonal: getListForAccountHistoryPersonal,
};

export default (
  view: ListView,
  call: Call,
  getRecordLink: GetRecordLink,
  categorization?: Categorization,
  activityData?: CallActivityData,
): FieldsList => {
  return gettersMap[view](call, getRecordLink, categorization, activityData);
};
