import { useMemo } from 'react';
import { types } from 'modules/categorization2';
import { normalize } from 'normalizr';
import { get as apiGet, baseCallApi, post } from 'api/common';
import { parseFromBackend } from 'modules/categorization';
import { config } from 'services/Config';
import { EType, ETypeString } from 'types/entities';
import { CategorizationValue } from './Categorization.types';
import { useCategorizationChange } from './useCategorizationChange';
import { rootSchema } from './schemas';
import { getInitialCategoryIds, createValueTreeFromIds, getDeepResults } from './utils';

const comType2Description = {
  YcCall: 'Звонок',
  Mail: 'Письмо',
};

export const useCategorizationHandlers = ({
  issueId,
  eId,
  eType,
  initialValue,
  targetDescription,
}: {
  issueId: number;
  eId: number;
  eType: string;
  initialValue?: CategorizationValue;
  targetDescription?: string;
}) => {
  const handleChange = useCategorizationChange({
    issueId,
    eId,
    eType,
  });
  const formattedInitialValue = useMemo(() => parseFromBackend(initialValue), [initialValue]);

  const handleLoad = (): Promise<types.CategoriesObject> => {
    return apiGet({
      url: '/view/category/tree',
      global: false,
      data: {
        issueId,
        workplace: eType,
        eId,
      },
    }).then((data) => {
      const normalizedData = normalize(data, rootSchema);
      const initialCategoryIds = formattedInitialValue
        ? getInitialCategoryIds(formattedInitialValue)
        : [];
      const byId = normalizedData.entities.items || {};
      const root = normalizedData.result.items || [];

      return {
        byId,
        root,
        highlighted: [],
        valueAsTree: createValueTreeFromIds(root, byId, initialCategoryIds),
      };
    });
  };

  const handleTipLoad = (id: number) => {
    return baseCallApi({
      url: `/view/category/${id}/daasTip/${eType}/${eId}`,
      global: false,
      showBackendErrorForUser: false,
      headers: {
        Accept: 'text/html',
      },
    });
  };

  const handleAlertsLoad = (categoryIds: number[]) => {
    return apiGet({
      url: '/alerts/active',
      global: false,
      data: {
        categoryIds,
      },
    });
  };

  const handleSearch = (
    text: string,
  ): Promise<{
    resultIds: number[];
    highlightRangesById: types.TextHighlightingById;
  }> => {
    return apiGet({
      url: '/view/category/tree',
      global: false,
      data: {
        issueId,
        workplace: eType,
        eId,
        text,
      },
    }).then((data) => {
      const normalizedData = normalize(data, rootSchema);
      const byId = normalizedData.entities.items || {};
      const ids = normalizedData.result.items || [];
      const highlightRangesById = Object.keys(byId).reduce(
        (acc, id) => ({
          ...acc,
          [id]: byId[id].highlightRanges,
        }),
        {},
      );

      return {
        resultIds: getDeepResults(ids, byId),
        highlightRangesById,
      };
    });
  };

  const handleTipComment = (payload: types.CommentFormExternalHandlerPayload) => {
    const { id, text } = payload;
    return post({
      url: '/actions/comment',
      global: false,
      data: {
        eType: EType.Category,
        eId: id,
        text,
      },
    });
  };

  return {
    onTipComment: handleTipComment,
    onSearch: handleSearch,
    onAlertsLoad: handleAlertsLoad,
    onTipLoad: handleTipLoad,
    onLoad: handleLoad,
    onChange: handleChange,
    initialValue: formattedInitialValue,
    targetMeta: {
      type: ETypeString[eType],
      id: eId,
      description: targetDescription || comType2Description[eType] || '',
    },
  };
};
