// TODO: https://st.yandex-team.ru/CRM-10202
import * as React from 'react';
import { connect } from 'react-redux';
import { withReduxContext, InjectedReduxContextProps } from 'modules/issues/redux';
import Access from 'utils/Access';
import { config } from 'services/Config';
import { Comment } from 'modules/startrek';
import Icon from 'lego/components/Icon';
import CommentForm from './CommentForm';
import * as TYPES from './types';
import Item, { Action } from './Item';

interface Props {
  comment: TYPES.Comment;
  access: TYPES.CommentProps;
  onRemove: () => void;
  onReply?: (text: string) => void;
  onEdit?: (comment: TYPES.Comment) => void;
  redux: {
    name: string;
  };
  issueId: number;
  timelineId?: string;
  nodeId: string;
}

interface State {
  isEdit: boolean;
}

class CommentIssue extends React.Component<Props, State> {
  public state = {
    isEdit: false,
  };

  private edit = (): void => {
    if (config.value.features.issueTimelineV2) {
      const { onEdit, comment } = this.props;

      if (typeof onEdit === 'function') {
        onEdit(comment);
      }
    } else {
      this.setState({ isEdit: true });
    }
  };

  private hide = (): void => {
    this.setState({ isEdit: false });
  };

  private handleReply = (): void => {
    const { onReply, comment } = this.props;

    if (comment.text && typeof onReply === 'function') {
      onReply(comment.text);
    }
  };

  public render(): React.ReactElement {
    const { comment, access, onRemove, issueId, ...other } = this.props;
    const { author, dt, invitees } = comment;

    const actions: Action[] = [];

    if (Access.isEdit(access.canReply)) {
      actions.push({
        key: 'reply',
        onClick: this.handleReply,
        children: 'Ответить',
      });
    }

    if (Access.isEdit(access.access)) {
      actions.push({
        key: 'edit',
        onClick: this.edit,
        children: 'Редактировать',
      });
    }

    if (Access.isEdit(access.access)) {
      actions.push({
        key: 'remove',
        onClick: onRemove,
        children: 'Удалить',
      });
    }

    return (
      <Item
        author={author}
        actions={this.state.isEdit ? undefined : actions}
        date={dt}
        icon={<Icon pack={Icon.PACKS.MATERIAL} icon="comment" />}
        afterAuthorName={<Comment.Invitees invitees={invitees} />}
        issueId={issueId}
        {...other}
      >
        {this.state.isEdit ? (
          <CommentForm
            objId={comment.id}
            filesObjectName="ocomment"
            cancelButton
            showSendAndOpenButton={Access.isEdit(access.canSendAndOpen)}
            onCancelClick={this.hide}
            onSuccess={this.hide}
            isPreventUnload
            initialValues={{
              issueId,
              text: comment.text,
              files: comment.files,
              invitees: comment.invitees,
            }}
            okButtonLabel="Сохранить"
            cancelButtonLabel="Отменить"
            focusOnMount
          />
        ) : (
          <Comment comment={comment} contentOnly />
        )}
      </Item>
    );
  }
}

interface ConnectedState {
  comment: TYPES.Comment;
  access: TYPES.CommentProps;
}

interface ConnectedDispatch {
  onRemove: () => void;
}

interface OwnProps {
  id: number;
  issueId: number;
  access?: TYPES.CommentProps;
}

export default withReduxContext(
  connect<ConnectedState, ConnectedDispatch, OwnProps & InjectedReduxContextProps>(
    (state, props) => {
      const data = props.redux.selectors.storage.comments.getItem(state, props.id) || {};

      return {
        comment: data.data,
        access: props.access || data.props,
      };
    },
    (dispatch, props) => ({
      onRemove: () => {
        if (window.confirm('Вы уверены, что хотите удалить этот комментарий?')) {
          dispatch(
            props.redux.slices.timelineSlice.asyncActions.commentDelete({
              commentId: props.id,
              issueId: props.issueId,
            }),
          );
        }
      },
    }),
  )(CommentIssue),
);
