// TODO: https://st.yandex-team.ru/CRM-10202

import * as React from 'react';
import { Tabs, Tab, ReactTabElement, ThemeType } from 'lego/components/Tabs';
import cx from 'classnames';
import { config } from 'services/Config';
import { MailForm } from '../Mail';
import CommentForm from '../CommentForm';
import MessageForm from '../MessageForm';
import InternalMessageForm from '../InternalMessageForm';
import css from './Communication.module.css';
import {
  CommunicationProps,
  CommunicationState,
  CommunicationTypes,
  InputRef,
} from './Communication.types';
import * as TYPES from '../types';
import { CONFIRM_MESSAGE, TYPE_MAPPER, COMTYPE_MAPPER } from './Communication.constants';

class Communication extends React.Component<CommunicationProps, CommunicationState> {
  constructor(props) {
    super(props);

    this.state = {
      tab: config.value.features.issueTimelineV2 ? COMTYPE_MAPPER[props.comType] : undefined,
      loadMailData: {},
      mailKey: 0,
      pristine: true,
      comment: undefined,
      message: undefined,
      internalMessage: undefined,
    };
  }

  private ref = React.createRef<InputRef>();

  private handleChangeTab = (value: CommunicationTypes | undefined) => {
    if (!this.canChangeTab()) {
      return;
    }

    this.resetCommunicationState();
    this.setState({ tab: value });
  };

  private resetCommunicationState = () => {
    this.setState((state) => ({
      tab: undefined,
      pristine: true,
      loadMailData: {},
      mailKey: state.mailKey + 1,
      comment: undefined,
      message: undefined,
      internalMessage: undefined,
    }));
  };

  private handleClearEditComment = () => {
    this.setState({ comment: undefined });
  };

  private handleClearEditMessage = () => {
    this.setState({ message: undefined });
  };

  private handleClearEditInternalMessage = () => {
    this.setState({ internalMessage: undefined });
  };

  private canChangeTab() {
    if (this.state.tab === CommunicationTypes.Mail || this.state.pristine) {
      return true;
    }

    return window.confirm(CONFIRM_MESSAGE);
  }

  private handleChangePristine = (pristine: boolean) => {
    this.setState({ pristine });
  };

  public editComment(comment: TYPES.Comment): void {
    if (!this.canChangeTab()) {
      return;
    }

    this.setState({ tab: CommunicationTypes.Comment, pristine: true, comment });
  }

  public editMessage(message: TYPES.ChatMessage): void {
    if (!this.canChangeTab()) {
      return;
    }

    this.setState({ tab: CommunicationTypes.Message, pristine: true, message });
  }

  public editInternalMessage(internalMessage: TYPES.ChatMessage): void {
    if (!this.canChangeTab()) {
      return;
    }

    this.setState({ tab: CommunicationTypes.InternalMessage, pristine: true, internalMessage });
  }

  public replyByComment(text): void {
    if (!this.canChangeTab()) {
      return;
    }

    this.setState({ tab: CommunicationTypes.Comment, pristine: true }, () => {
      requestAnimationFrame(() => {
        if (this.ref.current) {
          this.ref.current.reply(text);
        }
      });
    });
  }

  public replyByMail(mailId: number, type: string): void {
    if (!this.canChangeTab()) {
      return;
    }

    const { mailKey } = this.state;

    this.setState({
      tab: CommunicationTypes.Mail,
      loadMailData: TYPE_MAPPER[type](mailId),
      mailKey: mailKey + 1,
      pristine: true,
    });
  }

  public render(): React.ReactElement {
    const {
      issueId,
      className,
      showCommentForm,
      showMailForm,
      showMessageForm,
      showInternalMessageForm,
      canCommentSendAndOpen,
    } = this.props;
    const { tab, mailKey, loadMailData, comment, message, internalMessage } = this.state;

    const tabs: ReactTabElement[] = [];

    if (showCommentForm) {
      tabs.push(
        <Tab title="Написать комментарий" value={CommunicationTypes.Comment}>
          <CommentForm
            key={comment?.id || 'new_comment'}
            issueId={issueId}
            refTextComponent={this.ref}
            className={css.b__comment}
            onSuccess={this.resetCommunicationState}
            useLocationChangeAutoSave
            isPreventUnload
            onChangePristine={this.handleChangePristine}
            focusOnMount
            {...(comment
              ? {
                  objId: comment.id,
                  filesObjectName: 'ocomment',
                  cancelButton: true,
                  editInfoText: comment.text,
                  initialValues: {
                    text: comment.text,
                    files: comment.files,
                    invitees: comment.invitees,
                  },
                  okButtonLabel: 'Сохранить',
                  cancelButtonLabel: 'Отменить',
                  onClearEditClick: this.handleClearEditComment,
                  onCancelClick: this.resetCommunicationState,
                }
              : {
                  showSendAndOpenButton: canCommentSendAndOpen,
                })}
          />
        </Tab>,
      );
    }

    if (showMailForm) {
      tabs.push(
        <Tab title="Написать письмо" value={CommunicationTypes.Mail}>
          <MailForm
            key={mailKey}
            classNameBody={css.b__body}
            issueId={issueId}
            onSubmitSuccess={this.resetCommunicationState}
            loadMailData={loadMailData}
            onChangePristine={this.handleChangePristine}
          />
        </Tab>,
      );
    }

    if (showMessageForm) {
      tabs.push(
        <Tab title="Написать сообщение" value={CommunicationTypes.Message}>
          <MessageForm
            key={message?.id || 'new_message'}
            issueId={issueId}
            placeholder="Сообщение"
            refTextComponent={this.ref}
            className={css.b__comment}
            onSuccess={this.resetCommunicationState}
            useLocationChangeAutoSave
            isPreventUnload
            onChangePristine={this.handleChangePristine}
            focusOnMount
            {...(message
              ? {
                  objId: message.id,
                  filesObjectName: 'chatmessage',
                  cancelButton: true,
                  editInfoText: message.text,
                  initialValues: {
                    text: message.text,
                    files: message.files,
                  },
                  okButtonLabel: 'Сохранить',
                  cancelButtonLabel: 'Отменить',
                  onClearEditClick: this.handleClearEditMessage,
                  onCancelClick: this.resetCommunicationState,
                }
              : {})}
          />
        </Tab>,
      );
    }

    if (showInternalMessageForm) {
      tabs.push(
        <Tab title="Написать внутреннее сообщение" value={CommunicationTypes.InternalMessage}>
          <InternalMessageForm
            key={internalMessage?.id || 'new_internalmessage'}
            issueId={issueId}
            refTextComponent={this.ref}
            className={css.b__comment}
            onSuccess={this.resetCommunicationState}
            useLocationChangeAutoSave
            isPreventUnload
            onChangePristine={this.handleChangePristine}
            focusOnMount
            {...(internalMessage
              ? {
                  objId: internalMessage.id,
                  filesObjectName: 'internalchatmessage',
                  cancelButton: true,
                  editInfoText: internalMessage.text,
                  initialValues: {
                    text: internalMessage.text,
                    files: internalMessage.files,
                  },
                  okButtonLabel: 'Сохранить',
                  cancelButtonLabel: 'Отменить',
                  onClearEditClick: this.handleClearEditInternalMessage,
                  onCancelClick: this.resetCommunicationState,
                }
              : {})}
          />
        </Tab>,
      );
    }

    return (
      <Tabs
        value={tab}
        onChange={this.handleChangeTab}
        navigationTheme="start"
        className={cx(css.b, className)}
        theme={ThemeType.Top}
        canHide
      >
        {tabs}
      </Tabs>
    );
  }
}

export default Communication;
